/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.auditlog;

import com.sap.cds.CdsData;
import com.sap.cds.Struct;
import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.CdsModelReader;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.auditlog.Action;
import com.sap.cds.services.auditlog.AuditLogService;
import com.sap.cds.services.auditlog.ConfigChangeLog;
import com.sap.cds.services.auditlog.ConfigChangeLogContext;
import com.sap.cds.services.auditlog.DataAccessLog;
import com.sap.cds.services.auditlog.DataAccessLogContext;
import com.sap.cds.services.auditlog.DataModificationLog;
import com.sap.cds.services.auditlog.DataModificationLogContext;
import com.sap.cds.services.auditlog.SecurityLog;
import com.sap.cds.services.auditlog.SecurityLogContext;
import com.sap.cds.services.auditlog.event.TenantOffboardedEventContext;
import com.sap.cds.services.auditlog.event.TenantOnboardedEventContext;
import com.sap.cds.services.auditlog.event.UnauthorizedRequestEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.auditlog.events.TenantOffboardedEvent;
import com.sap.cds.services.impl.auditlog.events.TenantOnboardedEvent;
import com.sap.cds.services.impl.auditlog.events.security.UnauthorizedRequestEvent;
import com.sap.cds.services.outbox.OutboxMessage;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.outbox.StoredRequestContext;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.outbox.OutboxUtils;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Map;

/** An {@link EventHandler} to handle auditlog outbox related events. */
@ServiceName(value = "*", type = OutboxService.class)
public class AuditLogOutboxHandler implements EventHandler {

  private static final String USER = "user";
  private static final String TENANT = "tenant";
  private static final String LOGON_NAME = "logonName";
  private static final String IS_AUTHENTICATED = "isAuthenticated";
  private static final String IS_SYSTEM_USER = "isSystemUser";

  // common properties from auditlog-related contexts
  private static final String DATA = "data";
  private static final String CREATED_AT = "createdAt";
  private static final String ACTION = "action";
  private static final String MODIFICATIONS = "modifications";

  private CdsModel auditlogModel;

  public AuditLogOutboxHandler() {
    // interim solution
    String csnPath = "com/sap/cds/auditlog.csn";
    try (InputStream is =
        AuditLogOutboxHandler.class.getClassLoader().getResourceAsStream(csnPath)) {
      if (is == null) {
        throw new ErrorStatusException(CdsErrorStatuses.NO_AUDITLOG_MODEL, csnPath);
      }
      auditlogModel = CdsModelReader.read(is);
    } catch (IOException e) {
      throw new ErrorStatusException(CdsErrorStatuses.AUDITLOG_NOT_READABLE, csnPath, e);
    }
  }

  @On(event = AuditLogService.DEFAULT_NAME)
  private void publishedByOutbox(OutboxMessageEventContext context) {
    Service auditlogService =
        OutboxService.unboxed(context.getServiceCatalog().getService(AuditLogService.DEFAULT_NAME));

    // emit in the user context
    RequestContextRunner runner = context.getCdsRuntime().requestContext();
    runner
        .modifyUser(
            u -> {
              RequestContextAccessor requestContextAccessor;
              StoredRequestContext storedRequestContext =
                  context.getMessage().getStoredRequestContext();
              if (storedRequestContext != null) {
                requestContextAccessor =
                    new RequestContextAccessor(storedRequestContext, context.getEvent());
              } else {
                // the outbox entry seems to be in the old format, so we have to use the old way to
                // get the user information
                requestContextAccessor =
                    new RequestContextAccessor(context.getMessage(), context.getEvent());
                // always use tenant provided by auditlog message creation moment
                u.setTenant(requestContextAccessor.tenant);
              }
              u.setName(requestContextAccessor.user);
              u.setIsAuthenticated(requestContextAccessor.isAuthenticated);
              u.setIsSystemUser(requestContextAccessor.isSystemUser);
              u.setAdditionalAttribute(LOGON_NAME, requestContextAccessor.logonName);
            })
        .run(
            req -> {
              EventContextAccessor contextAccessor =
                  new EventContextAccessor(context, auditlogModel);
              auditlogService.emit(contextAccessor.context);
            });
    context.setCompleted();
  }

  @Before(event = AuditLogService.DEFAULT_NAME)
  private void prepareOutboxMessage(OutboxMessageEventContext context) {
    // prepare outbox message for storage only
    if (Boolean.TRUE.equals(context.getIsInbound())) {
      return;
    }

    OutboxMessage message = context.getMessage();
    try {
      // serialize request context information
      RequestContext requestContext = RequestContext.getCurrent(context.getCdsRuntime());
      StoredRequestContext storedRequestContext = context.getMessage().getStoredRequestContext();
      if (storedRequestContext == null) {
        storedRequestContext = StoredRequestContext.create();
        message.setStoredRequestContext(storedRequestContext);
      }
      storedRequestContext.put(USER, requestContext.getUserInfo().getName());
      storedRequestContext.put(IS_AUTHENTICATED, requestContext.getUserInfo().isAuthenticated());
      storedRequestContext.put(IS_SYSTEM_USER, requestContext.getUserInfo().isSystemUser());
      String logonName = (String) requestContext.getUserInfo().getAdditionalAttribute(LOGON_NAME);
      if (!StringUtils.isEmpty(logonName)) {
        storedRequestContext.put(LOGON_NAME, logonName);
      }

      // post-process event context parameters
      CdsData data = (CdsData) message.getParams().get(DATA);
      message.getParams().clear();
      message.getParams().put(DATA, data.toJson());
    } catch (Exception e) {
      throw new ErrorStatusException(
          CdsErrorStatuses.CONTEXT_SERIALIZATION_FAILED, context.getEvent(), e);
    }
  }

  private static class EventContextAccessor {

    private final EventContext context;

    private EventContextAccessor(OutboxMessageEventContext outboxContext, CdsModel auditlogModel) {
      try {
        // restore event context parameters
        this.context = OutboxUtils.toEventContext(outboxContext);
        OutboxMessage message = outboxContext.getMessage();
        CdsStructuredType type =
            auditlogModel.getEvent("com.sap.cds.services.auditlog." + context.getEvent());
        String dataJson =
            message.getParams() == null ? (String) message.get(DATA) : (String) context.get(DATA);
        Map<String, Object> data = StructDataParser.create(type).parseObject(dataJson);
        adaptActionType(data, context.getEvent());

        setData(context, context.getEvent(), data);
        context.put(CREATED_AT, outboxContext.getTimestamp());
      } catch (Exception e) {
        throw new ErrorStatusException(
            CdsErrorStatuses.CONTEXT_DESERIALIZATION_FAILED, outboxContext.getEvent(), e);
      }
    }

    private void setData(EventContext context, String event, Map<String, Object> data) {
      // set data using the typed interface according to the concrete event type
      if (event.equals(DataModificationLogContext.CDS_NAME)) {
        DataModificationLog log = Struct.access(data).as(DataModificationLog.class);
        context.as(DataModificationLogContext.class).setData(log);
      } else if (event.equals(ConfigChangeLogContext.CDS_NAME)) {
        ConfigChangeLog log = Struct.access(data).as(ConfigChangeLog.class);
        context.as(ConfigChangeLogContext.class).setData(log);
      } else if (event.equals(SecurityLogContext.CDS_NAME)) {
        SecurityLog log = Struct.access(data).as(SecurityLog.class);
        context.as(SecurityLogContext.class).setData(log);
      } else if (event.equals(DataAccessLogContext.CDS_NAME)) {
        DataAccessLog log = Struct.access(data).as(DataAccessLog.class);
        context.as(DataAccessLogContext.class).setData(log);
      } else if (event.equals(TenantOnboardedEventContext.CDS_NAME)) {
        TenantOnboardedEvent log = Struct.access(data).as(TenantOnboardedEvent.class);
        context.put(DATA, log); // TODO
      } else if (event.equals(TenantOffboardedEventContext.CDS_NAME)) {
        TenantOffboardedEvent log = Struct.access(data).as(TenantOffboardedEvent.class);
        context.put(DATA, log); // TODO
      } else if (event.equals(UnauthorizedRequestEventContext.CDS_NAME)) {
        UnauthorizedRequestEvent log = Struct.access(data).as(UnauthorizedRequestEvent.class);
        context.put(DATA, log); // TODO
      }
    }

    @SuppressWarnings("unchecked")
    private void adaptActionType(Map<String, Object> data, String event) {
      // adapt the CDS types
      if (event.equals(DataModificationLogContext.CDS_NAME)) {
        Collection<Map<String, Object>> modifications =
            (Collection<Map<String, Object>>) ((Map<String, ?>) data).get(MODIFICATIONS);
        if (modifications != null) {
          modifications.forEach(
              modification -> {
                Object action = modification.get(ACTION);
                if (action != null) {
                  modification.put(ACTION, Action.valueOf(action.toString()));
                }
              });
        }
      } else if (event.equals(ConfigChangeLogContext.CDS_NAME)) {
        Object action = data.get(ACTION);
        if (action != null) {
          data.put(ACTION, Action.valueOf(action.toString()));
        }
      }
    }
  }

  private static class RequestContextAccessor {

    private final String user;
    private final String tenant;
    private final String logonName;
    private final boolean isAuthenticated;
    private final boolean isSystemUser;

    public RequestContextAccessor(Map<String, Object> map, String outboxEvent) {
      try {
        // restore request context
        this.user = (String) map.get(USER);
        this.tenant = (String) map.get(TENANT);
        this.logonName = (String) map.get(LOGON_NAME);
        this.isAuthenticated = Boolean.TRUE.equals(map.get(IS_AUTHENTICATED));
        this.isSystemUser = Boolean.TRUE.equals(map.get(IS_SYSTEM_USER));
      } catch (Exception e) {
        throw new ErrorStatusException(
            CdsErrorStatuses.CONTEXT_DESERIALIZATION_FAILED, outboxEvent, e);
      }
    }
  }
}
