/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.auditlog;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.TenantAwareCache;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.lang3.StringUtils;

/** This class provides access to personal data caches. */
class PersonalDataCaches {

  private final TenantAwareCache<Map<String, PersonalDataMeta>, CdsModel> metaCache;
  private final TenantAwareCache<Map<String, PersonalDataUtils>, CdsModel> utilsCache;

  PersonalDataCaches(CdsRuntime runtime) {
    this.metaCache = TenantAwareCache.create(ConcurrentHashMap::new, runtime);
    this.utilsCache = TenantAwareCache.create(ConcurrentHashMap::new, runtime);
  }

  /**
   * Gets a {@link PersonalDataMeta} for the given {@link CdsStructuredType} from the cache.
   *
   * @param type the {@link CdsStructuredType}
   * @return the meta data for given type
   */
  PersonalDataMeta getMeta(CdsStructuredType type) {
    if (StringUtils.isNotEmpty(type.getQualifiedName())) {
      return this.metaCache
          .findOrCreate()
          .computeIfAbsent(type.getQualifiedName(), key -> new PersonalDataMeta(type));
    }
    return null;
  }

  /**
   * Returns a {@link PersonalDataUtils} from the cache.
   *
   * @param type the {@link CdsStructuredType}
   * @return the {@link PersonalDataUtils} for given type
   */
  PersonalDataUtils getUtils(CdsStructuredType type) {
    if (StringUtils.isNotEmpty(type.getQualifiedName())) {
      return this.utilsCache
          .findOrCreate()
          .computeIfAbsent(type.getQualifiedName(), key -> new PersonalDataUtils(type, this));
    }
    return null;
  }
}
