/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.auditlog;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Expand;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.impl.utils.TargetAwareCqnModifier;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;

class PersonalDataModifier extends TargetAwareCqnModifier {

  private final List<PersonalDataModifier> children = new LinkedList<>();
  private boolean innerCqn;
  private final boolean addPersonalData;
  private final PersonalDataCaches caches;
  private final boolean isDraftShadowEntity;

  /** Stores the accessed sensitive elements of all accessed entities */
  private final Map<String, Pair<CdsStructuredType, Map<String, CqnElementRef>>> sensitiveElements =
      new HashMap<>();

  /**
   * @param target the target entity
   * @param addPersonalData if {@code true}, also elements annotated with
   *     {@code @PersonalData.IsPotentiallyPersonal} are considered
   * @param caches caches for personal data meta and utils
   */
  PersonalDataModifier(CdsEntity target, boolean addPersonalData, PersonalDataCaches caches) {
    this(target, false, addPersonalData, caches);
  }

  private PersonalDataModifier(
      CdsEntity target, boolean innerCqn, boolean addPersonalData, PersonalDataCaches caches) {
    super(Objects.requireNonNull(target, "target must not null"));
    this.innerCqn = innerCqn;
    this.addPersonalData = addPersonalData;
    this.caches = Objects.requireNonNull(caches, "caches must not null");
    this.isDraftShadowEntity = target.getQualifiedName().endsWith("_drafts");
  }

  @Override
  public CqnStructuredTypeRef ref(CqnStructuredTypeRef ref) {
    // we have reached inner cqn statement here.
    // ref() is guaranteed to be called before where() within a statement.
    this.innerCqn = true;
    return ref;
  }

  @Override
  protected TargetAwareCqnModifier create(CdsEntity target) {
    PersonalDataModifier result =
        new PersonalDataModifier(target, this.innerCqn, this.addPersonalData, this.caches);
    this.children.add(result);
    return result;
  }

  @Override
  public List<CqnSelectListItem> selectAll() {
    if (!isDraftShadowEntity) {
      // adds sensitive fields to selected items
      getTarget()
          .elements()
          .filter(element -> element.getType().isSimple() || element.getType().isArrayed())
          .forEach(element -> addIfSensitive(element, CQL.get(element.getName())));
    }
    return super.selectAll();
  }

  @Override
  public CqnValue ref(CqnElementRef ref) {
    if (this.innerCqn && !isDraftShadowEntity) {
      CdsModelUtils.findElement(getTarget(), ref)
          .ifPresent(element -> addIfSensitive(element, ref));
    }
    return ref;
  }

  private void addIfSensitive(CdsElement element, CqnElementRef ref) {
    if ((CdsAnnotations.PERSONALDATA_POTENTIALLYSENSITIVE.isTrue(element)
        || (CdsAnnotations.PERSONALDATA_POTENTIALLYPERSONAL.isTrue(element)
            && this.addPersonalData))) {

      // select keys and data subject id
      CdsStructuredType def = element.getDeclaringType();
      String structureKey = def.getQualifiedName();
      String elementKey = ref.path();

      if (this.sensitiveElements.containsKey(structureKey)) {
        Map<String, CqnElementRef> refs = this.sensitiveElements.get(structureKey).getRight();
        // avoid duplicate entries in sensitiveElements
        refs.putIfAbsent(elementKey, ref);
      } else {
        this.sensitiveElements.put(
            structureKey, Pair.of(def, new HashMap<>(Map.of(elementKey, ref))));
      }
    }
  }

  @Override
  public List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
    if (this.innerCqn) {
      this.sensitiveElements.forEach(
          (key, value) -> {
            // resolve the keys and data subject id for the accessed entity
            PersonalDataMeta pdMeta = getMeta(value.getLeft());

            // TODO check if key is already in select list and don't add it again
            Set<CqnSelectListValue> elements =
                pdMeta.getKeyNames().stream()
                    .map(str -> CQL.get(str).as(PersonalDataUtils.getKeyAlias(str)))
                    .collect(Collectors.toSet());
            elements.addAll(getUtils(value.getLeft()).resolveDataSubjectId());

            // The type with sensitive data is reached via paths from other entities
            if (value.getRight().values().stream().anyMatch(r -> r.size() > 1)) {
              boolean marked = false;
              for (CqnElementRef r : value.getRight().values()) {
                if (r.size() == 1 && !marked) {
                  items.addAll(elements);
                  marked = true; // To avoid second pass if paths and items are mixed together
                } else {
                  // TODO: add alias to expand
                  // TODO: figure out way to optimize this
                  // Refs might reach out to other types several times
                  // and/or return back to the type
                  items.add(expandPath(r, elements));
                }
              }
            } else {
              // All refs have one segment. This is either root or an expand
              // (TargetAwareCqnModifier visits expands)
              items.addAll(elements);
            }
          });
      this.innerCqn = false; // ignore refs in groupBy/orderBy
    }
    return super.items(items);
  }

  private Expand<?> expandPath(CqnElementRef ref, Set<CqnSelectListValue> elements) {
    // Skip target segment of the ref
    ListIterator<Segment> iterator = ref.segments().listIterator(ref.size() - 1);
    Expand<?> expand = CQL.to(iterator.previous().id()).expand(elements);
    while (iterator.hasPrevious()) {
      Segment s = iterator.previous();
      expand = CQL.to(s.id()).expand(expand);
    }
    return expand;
  }

  Map<String, Pair<CdsStructuredType, Map<String, CqnElementRef>>> getSensitiveElements() {
    Map<String, Pair<CdsStructuredType, Map<String, CqnElementRef>>> result =
        new HashMap<>(this.sensitiveElements);
    // traverse into all children
    this.children.forEach(
        child -> {
          Map<String, Pair<CdsStructuredType, Map<String, CqnElementRef>>> sensitiveElementsChild =
              child.getSensitiveElements();
          sensitiveElementsChild.forEach(
              (key, value) -> {
                if (result.containsKey(key)) {
                  value
                      .getRight()
                      .forEach(
                          (s, cqnElementRef) ->
                              result.get(key).getRight().computeIfAbsent(s, v -> cqnElementRef));
                } else {
                  result.put(key, value);
                }
              });
        });
    return result;
  }

  private PersonalDataUtils getUtils(CdsStructuredType entity) {
    return this.caches.getUtils(entity);
  }

  private PersonalDataMeta getMeta(CdsStructuredType entity) {
    return this.caches.getMeta(entity);
  }
}
