/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.authorization;

import com.sap.cds.impl.builder.model.Conjunction;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.sap.cds.util.CdsModelUtils;
import java.util.Set;

// This is temporary workaround to convert conditions coming from restrict predicates containing
// path expressions
// until this support is implemented in cds4j.
public class PathExpressionPredicateConverter {

  private static final StructuredType<?> $OUTER = CQL.to(CqnExistsSubquery.OUTER);

  private PathExpressionPredicateConverter() {
    // hidden
  }

  public static CqnPredicate convert(
      CdsStructuredType target, CqnPredicate authorizationCondition) {
    return convert(target, null, authorizationCondition);
  }

  static CqnPredicate convert(
      CdsStructuredType target, CdsElement association, CqnPredicate authorizationCondition) {
    if (authorizationCondition.equals(CQL.FALSE)
        || authorizationCondition.asPredicate() instanceof CqnExistsSubquery) {
      return authorizationCondition;
    }

    // Move auth condition with paths into where exists subquery, which also works in infix filters
    // (deep auth)
    // subquery target entity is outer target entity (correlated via keys: k1 = $outer.k1 and k2 =
    // $outer.k2)
    // the authorization condition is added to the where condition of the subquery
    HasRefsWithComplexPaths v = new HasRefsWithComplexPaths();
    authorizationCondition.accept(v);
    if (v.containsPaths()) {
      // Either keys that are correlated to outer (for parent) or self-join
      Set<String> keys =
          association != null
              ? CdsModelUtils.targetKeys(association)
              : CdsModelUtils.concreteKeyNames(target);
      CqnPredicate joinCondition =
          keys.stream().map(k -> CQL.get(k).eq($OUTER.get(k))).collect(Conjunction.and());
      authorizationCondition = CQL.and(joinCondition, authorizationCondition);
      return CQL.exists(Select.from(target.getQualifiedName()).where(authorizationCondition));
    }
    return authorizationCondition;
  }

  private static class HasRefsWithComplexPaths implements CqnVisitor {

    private boolean hasPath = false;

    @Override
    public void visit(CqnMatchPredicate match) {
      if (!hasPath) {
        hasPath = isPath(match.ref());
      }
    }

    @Override
    public void visit(CqnElementRef elementRef) {
      if (!hasPath) {
        hasPath = isPath(elementRef);
      }
    }

    private static boolean isPath(CqnReference ref) {
      return ref.size() > 1 && !CdsConstants.$USER.equals(ref.firstSegment());
    }

    public boolean containsPaths() {
      return hasPath;
    }
  }
}
