/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.authorization;

import static com.sap.cds.services.impl.authorization.ReferenceAssociationTraverser.NO_COMPOSITIONS;
import static com.sap.cds.services.impl.authorization.ReferenceAssociationTraverser.modify;
import static com.sap.cds.util.CqnStatementUtils.targetEntity;

import com.sap.cds.impl.builder.model.ExpandBuilder;
import com.sap.cds.impl.builder.model.StructuredTypeRefImpl;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnInSubquery;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.util.CdsModelUtils;
import java.util.List;
import java.util.Objects;

@SuppressWarnings("UnstableApiUsage")
public class ReadStatementAuthorizationModifier implements Modifier {

  private static final Predicate IS_ACTIVE_EQ_FALSE = CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false);

  private final AuthorizationService authorizationService;
  private final CdsModel model;
  private final String event;
  private final CdsStructuredType
      parentTarget; // target entity of a parent statement -> e.g. query in the source, needed only
  // for $outer
  private final CdsStructuredType target; // target of the current statement
  private boolean isInnerStatement = false;

  public ReadStatementAuthorizationModifier(
      AuthorizationService authService, CdsReadEventContext context) {
    this(
        authService,
        context.getModel(),
        context.getEvent(),
        null, // No parent on root call
        targetEntity(context.getCqn(), context.getModel()));
  }

  private ReadStatementAuthorizationModifier(
      AuthorizationService authorizationService,
      CdsModel model,
      String event,
      CdsStructuredType parentTarget,
      CdsStructuredType target) {
    this.authorizationService = Objects.requireNonNull(authorizationService);
    this.model = Objects.requireNonNull(model);
    this.event = event;
    this.target = target;
    this.parentTarget = parentTarget;
  }

  @Override
  public CqnStructuredTypeRef ref(CqnStructuredTypeRef ref) {
    // Source is always copied first
    isInnerStatement = true;
    Segment targetSegment = ref.targetSegment();

    CdsEntity root = model.getEntity(ref.firstSegment());
    List<Segment> segments =
        modify(
            root,
            ref,
            NO_COMPOSITIONS,
            segment -> {
              if (segment.segment() != targetSegment) {
                return secureSegment(segment);
              }
              return segment.segment();
            });
    return StructuredTypeRefImpl.typeRef(segments);
  }

  @Override
  public CqnPredicate where(Predicate where) {
    // false indicates that this statement is an outer statement
    if (isInnerStatement) {
      isInnerStatement = false;
      // join with where + auth OR IsActiveEntity=false
      CqnPredicate authorizationCondition =
          authorizationService.calcWhereCondition(target.getQualifiedName(), event);
      if (authorizationCondition != null) {
        return CQL.and(where, authorizedOrInactive(target, authorizationCondition));
      }
    }
    return where;
  }

  @Override
  public CqnPredicate exists(Select<?> subQuery) {
    return CQL.exists(CQL.copy(subQuery, newModifier(targetEntity(subQuery, model))));
  }

  @Override
  public CqnPredicate in(CqnInSubquery inSubquery) {
    CqnSelect subQuery = inSubquery.subquery();
    return CQL.in(
        inSubquery.value(), CQL.copy(subQuery, newModifier(targetEntity(subQuery, model))));
  }

  @Override
  public CqnPredicate match(CqnMatchPredicate match) {
    List<Segment> segments = copyRef(match.ref());
    return CQL.match(
        CQL.to(segments).as(match.ref().alias().orElse(null)).asRef(),
        match.predicate().orElse(null),
        match.quantifier());
  }

  @Override
  public CqnSelectListItem expand(CqnExpand expand) {
    rejectStarAndEmptyRefs(expand.ref());

    List<Segment> segments = copyRef(expand.ref());
    CdsStructuredType expandTarget = CdsModelUtils.target(target, expand.ref().segments());
    CqnExpand result =
        CQL.to(segments)
            .as(expand.ref().alias().orElse(null))
            .expand(ExpressionVisitor.copy(expand.items(), newModifier(expandTarget)))
            .limit(expand.top(), expand.skip())
            .inlineCount(expand.hasInlineCount())
            .orderBy(expand.orderBy())
            .as(expand.displayName());
    if (((ExpandBuilder<?>) expand).lazy()) {
      ((ExpandBuilder<?>) result).lazy(true);
    }
    return result;
  }

  @Override
  public CqnSelectListItem inline(CqnInline inline) {
    rejectStarAndEmptyRefs(inline.ref());

    List<Segment> segments = copyRef(inline.ref());
    CdsStructuredType inlineTarget = CdsModelUtils.target(target, inline.ref().segments());
    return CQL.to(segments)
        .as(inline.ref().alias().orElse(null))
        .inline(ExpressionVisitor.copy(inline.items(), newModifier(inlineTarget)));
  }

  @Override
  public CqnValue ref(CqnElementRef ref) {
    // ref with association as a single segment pulls FK only.
    if (!isPath(ref)) {
      return ref;
    }

    // Secure ref in the select list -> path access to something
    // while avoiding eager copies of the segments. In case of element refs
    // it could be that the elements, structures are far more common than
    // the associations
    List<Segment> segments;
    // Assume that $outer is always resolvable from the target of the parent type
    // E.g. `select from Order.item where exists $outer.active = true` parent target is item
    if (CqnExistsSubquery.OUTER.equals(ref.firstSegment())) {
      segments = modify(parentTarget, ref, NO_COMPOSITIONS, this::secureSegment);
    } else {
      segments = modify(target, ref, NO_COMPOSITIONS, this::secureSegment);
    }
    return CQL.get(segments).as(ref.alias().orElse(null));
  }

  private List<Segment> copyRef(CqnReference ref) {
    rejectStarAndEmptyRefs(ref);
    return modify(target, ref, NO_COMPOSITIONS, this::secureSegment);
  }

  private Segment secureSegment(ResolvedSegment current) {
    CqnPredicate authorizationCondition =
        authorizationService.calcWhereCondition(current.type().getQualifiedName(), event);
    if (authorizationCondition != null) {
      CqnPredicate rewritten =
          PathExpressionPredicateConverter.convert(
              current.type(), current.element(), authorizationCondition);
      CqnPredicate orInactive = authorizedOrInactive(current.type(), rewritten);
      return CQL.refSegment(
          current.segment().id(), CQL.and(current.segment().filter().orElse(null), orInactive));
    }
    return current.segment();
  }

  private CqnPredicate authorizedOrInactive(
      CdsStructuredType target, CqnPredicate authorizationCondition) {
    if (DraftUtils.isDraftEnabled(target)) {
      return CQL.or(authorizationCondition, IS_ACTIVE_EQ_FALSE);
    }
    return authorizationCondition;
  }

  private ReadStatementAuthorizationModifier newModifier(CdsStructuredType newTarget) {
    return new ReadStatementAuthorizationModifier(
        authorizationService, model, event, target, newTarget);
  }

  private static void rejectStarAndEmptyRefs(CqnReference ref) {
    // Defensive, star and empty refs are rejected earlier with a proper error message
    if (ref.segments().isEmpty() || ref.firstSegment().equals("*")) {
      throw new IllegalStateException(
          "The star expand or empty ref are not supposed to reach authorization check");
    }
  }

  private static boolean isPath(CqnElementRef ref) {
    return ref.size() > 1 && !CdsConstants.$USER.equals(ref.firstSegment());
  }
}
