/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_RANGE;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.ValidatorErrorUtils;
import com.sap.cds.services.impl.utils.ValidatorExecutor;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import java.util.stream.Collectors;

@ServiceName(value = "*", type = ApplicationService.class)
public class EnumAssertionHandler implements EventHandler {

  @Before
  @HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
  public void runCheck(EventContext context) {
    CdsDataProcessor cdsDataProcessor =
        CdsDataProcessor.create()
            .addValidator(
                EnumAssertionHandler::isEnumWithAssertRange,
                (path, element, elementValue) -> {
                  ensureValidEnum(context, path, element, elementValue);
                });
    ValidatorExecutor.processEntities(cdsDataProcessor, context);
  }

  private static void ensureValidEnum(
      EventContext context, Path path, CdsElement element, Object elementValue) {
    if (elementValue != null) {
      CdsEnumType<?> enumElement = element.getType();
      if (!enumElement.hasValue(elementValue)) {
        String validEnumValues =
            enumElement.enumerals().values().stream()
                .map(e -> e.value() + " (" + e.name() + ")")
                .collect(Collectors.joining(", "));
        ValidatorErrorUtils.handleValidationError(
            context,
            path,
            element,
            CdsAnnotations.ASSERT_RANGE_MESSAGE,
            CdsErrorStatuses.INVALID_ENUM,
            element.getName(),
            validEnumValues);
      }
    }
  }

  private static boolean isEnumWithAssertRange(Path path, CdsElement element, CdsType type) {
    return type.isEnum() && !Boolean.FALSE.equals(ASSERT_RANGE.getOrDefault(element));
  }
}
