/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_FORMAT;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.ValidatorErrorUtils;
import com.sap.cds.services.impl.utils.ValidatorExecutor;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.model.CdsAnnotations;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

@ServiceName(value = "*", type = ApplicationService.class)
public class FormatAssertionHandler implements EventHandler {

  // cache the regex pattern due to Performance of pattern compilation
  private final TenantAwareCache<Map<String, Pattern>, CdsModel> patternCache;

  public FormatAssertionHandler(CdsRuntime runtime) {
    this.patternCache = TenantAwareCache.create(ConcurrentHashMap::new, runtime);
  }

  @Before
  @HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
  public void runCheck(EventContext context) {
    CdsDataProcessor cdsDataProcessor =
        CdsDataProcessor.create()
            .addValidator(
                FormatAssertionHandler::isStringWithFormat,
                (path, element, value) -> {
                  validate(context, path, element, value);
                });
    ValidatorExecutor.processEntities(cdsDataProcessor, context);
  }

  private void validate(EventContext context, Path path, CdsElement element, Object value) {
    if (value != null) {
      // Throw an exception if the annotation value is not a valid pattern
      Object regex = ASSERT_FORMAT.getOrValue(element, null);
      Pattern pattern =
          getPattern(
              (String) regex, element.getName(), element.getDeclaringType().getQualifiedName());
      // Throw an exception if the element value does not match the pattern
      ensureValidFormat(context, value.toString(), pattern, path, element);
    }
  }

  private void ensureValidFormat(
      EventContext context,
      String elementConvertedValue,
      Pattern pattern,
      Path path,
      CdsElement element) {
    Matcher matcher = pattern.matcher(elementConvertedValue);
    if (!matcher.matches()) {
      ValidatorErrorUtils.handleValidationError(
          context,
          path,
          element,
          CdsAnnotations.ASSERT_FORMAT_MESSAGE,
          CdsErrorStatuses.INVALID_STRING_VALUE,
          element.getName());
    }
  }

  private static boolean isStringWithFormat(Path path, CdsElement e, CdsType t) {
    return t.isSimpleType(CdsBaseType.STRING)
        && (ASSERT_FORMAT.getOrValue(e, null) instanceof String);
  }

  private Pattern getPattern(String regex, String elementName, String entityName) {
    Pattern pattern = this.patternCache.findOrCreate().get(regex);
    if (pattern == null) {
      // Try to create cached Pattern
      try {
        pattern = Pattern.compile(regex);
      } catch (PatternSyntaxException e) {
        throw new ErrorStatusException(
            CdsErrorStatuses.INVALID_ANNOTATION,
            ASSERT_FORMAT.toString(),
            elementName,
            entityName,
            e);
      }
      this.patternCache.findOrCreate().put(regex, pattern);
    }
    return pattern;
  }
}
