/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CqnUtils.addWhere;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnFilterableStatement;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.authorization.PathExpressionPredicateConverter;
import com.sap.cds.services.impl.authorization.ReadStatementAuthorizationModifier;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CqnUtils;
import com.sap.cds.util.CqnStatementUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ServiceName(value = "*", type = ApplicationService.class)
public class InstanceBasedAuthorizationHandler implements EventHandler {

  private static final Logger logger =
      LoggerFactory.getLogger(InstanceBasedAuthorizationHandler.class);

  private final boolean authorizeDeeply;
  private final boolean regardCustomEvents;
  private final boolean checkInputData;

  public InstanceBasedAuthorizationHandler(CdsRuntime runtime) {
    var authorization =
        runtime.getEnvironment().getCdsProperties().getSecurity().getAuthorization();
    authorizeDeeply = authorization.getDeep().isEnabled();
    regardCustomEvents = authorization.getInstanceBased().getCustomEvents().isEnabled();
    checkInputData = authorization.getInstanceBased().getCheckInputData().isEnabled();
  }

  // Restriction annotations from underlying projected entities are added to the service entity by
  // the compiler provided that no restrictions are defined already.
  // So the authorization has to be tested only on service level.
  @Before(event = "*")
  @HandlerOrder(OrderConstants.Before.CHECK_INSTANCE_BASED_AUTHORIZATION)
  private void checkAuthorization(EventContext context) {
    // privileged users just skip authorization checks
    if (!context.getUserInfo().isPrivileged()) {
      extendStatements(context);
    }
  }

  private boolean isRelevantEvent(String eventName, String entityName) {
    return eventName.equals(CqnService.EVENT_DELETE)
        || eventName.equals(CqnService.EVENT_UPDATE)
        || eventName.equals(DraftService.EVENT_DRAFT_EDIT)
        || eventName.equals(DraftService.EVENT_DRAFT_SAVE)
        || (regardCustomEvents
            && !CdsModelUtils.isStandardCdsEvent(eventName)
            && entityName != null);
  }

  /**
   * Extend the event context's CQN statement with an instance based authorization condition if
   * available.
   */
  private void extendStatements(EventContext context) {
    AuthorizationService authService =
        context
            .getServiceCatalog()
            .getService(AuthorizationService.class, AuthorizationService.DEFAULT_NAME);

    final String event = context.getEvent();
    final String serviceName =
        ((ApplicationService) context.getService()).getDefinition().getQualifiedName();

    CdsEntity targetEntity = context.getTarget();
    String targetEntityName = null;
    if (targetEntity != null) {
      targetEntityName = targetEntity.getQualifiedName();
    }

    if (event.equals(CqnService.EVENT_READ)) {
      CdsReadEventContext cdsContext = context.as(CdsReadEventContext.class);
      if (authorizeDeeply) {
        CqnSelect secured =
            CQL.copy(
                cdsContext.getCqn(),
                new ReadStatementAuthorizationModifier(authService, cdsContext));
        if (logger.isDebugEnabled()) {
          logger.debug(
              "Statement is extended with security conditions for target '{}' of service '{}': '{}'",
              targetEntityName,
              serviceName,
              CqnStatementUtils.anonymizeStatement(secured));
        }
        cdsContext.setCqn(secured);
      } else {
        CqnSelect select = cdsContext.getCqn();
        CqnPredicate whereCondition =
            adaptWhereConditionForDraftIfNecessary(authService, event, targetEntity, false);
        if (whereCondition != null) {
          select = CqnUtils.addWhere(select, whereCondition);
        }
        cdsContext.setCqn(select);
      }
    } else if (isRelevantEvent(event, targetEntityName)) {
      // process instance-based authorization by appending the CQN where statements with additional
      // restrictions
      CqnPredicate whereCondition =
          adaptWhereConditionForDraftIfNecessary(authService, event, targetEntity, true);
      if (whereCondition != null) {
        if (context.get("cqn") instanceof CqnFilterableStatement cqn) {
          context.put("cqn", addWhere(cqn, whereCondition));
        }
        if (logger.isDebugEnabled()) {
          logger.debug(
              "Created CQN authorization predicate {} for entity '{}' of service '{}', event {} and user {}",
              whereCondition.toJson(),
              targetEntityName,
              serviceName,
              event,
              context.getUserInfo().getName());
        }
      } else if (logger.isDebugEnabled()) {
        logger.debug(
            "No CQN authorization predicate required for entity '{}' of service '{}', event {} and user {}",
            targetEntityName,
            serviceName,
            event,
            context.getUserInfo().getName());
      }
    }
    // all checks derived from the model passes here.
    logger.debug(
        "Instance-based authorization check passed for event '{}' on service '{}'",
        event,
        serviceName);
  }

  /**
   * Get the event's where-condition and adapt it if necessary for the given event. Adaption is
   * relevant for Drafts.
   */
  private CqnPredicate adaptWhereConditionForDraftIfNecessary(
      AuthorizationService authService,
      String eventName,
      CdsEntity targetEntity,
      boolean rewritePath) {
    if (targetEntity == null) {
      return authService.calcWhereCondition(null, eventName);
    }

    CqnPredicate whereCondition =
        authService.calcWhereCondition(targetEntity.getQualifiedName(), eventName);
    if (whereCondition == null) {
      return null;
    }

    // TODO: Remove this workaround when cds4j can do it
    if (rewritePath) { // This is not necessary for selects
      whereCondition = PathExpressionPredicateConverter.convert(targetEntity, whereCondition);
    }

    // legacy check to forbid activation of drafts with values outside of instance-based condition
    // this special handling is no longer required in case checkInputData is set
    boolean relaxInactive = checkInputData || !DraftService.EVENT_DRAFT_SAVE.equals(eventName);
    if (relaxInactive && DraftUtils.isDraftEnabled(targetEntity)) {
      return CQL.or(whereCondition, CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(false));
    }

    return whereCondition;
  }
}
