/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnFilterableStatement;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = ApplicationService.class)
public class InstanceBasedAuthorizationRejectionHandler implements EventHandler {
  private final boolean regardCustomEvents;

  public InstanceBasedAuthorizationRejectionHandler(CdsRuntime runtime) {
    var authorization =
        runtime.getEnvironment().getCdsProperties().getSecurity().getAuthorization();
    regardCustomEvents = authorization.getInstanceBased().getCustomEvents().isEnabled();
  }

  /**
   * Reject relevant events if they target a single entity which is restricted and whose
   * instance-based condition is not fulfilled.
   */
  @Before(event = "*")
  @HandlerOrder(OrderConstants.Before.CHECK_INSTANCE_BASED_AUTHORIZATION_REJECTION)
  private void checkAuthorization(EventContext context) {
    if (context.getUserInfo().isPrivileged()) {
      return; // privileged users just skip authorization checks
    }

    if (!isRelevantEventForRejectionHandler(context)) {
      return;
    }

    CdsEntity targetEntity = context.getTarget();
    String eventName = context.getEvent();
    String targetEntityName = targetEntity.getQualifiedName();

    AuthorizationService authService =
        context
            .getServiceCatalog()
            .getService(AuthorizationService.class, AuthorizationService.DEFAULT_NAME);
    CqnPredicate authFilter = authService.calcWhereCondition(targetEntityName, eventName);
    if (authFilter == null) {
      return;
    }

    CqnPredicate keyFilter = QueryAuthorizationCheckHelper.keyFilterFromStatement(context);
    if (keyFilter == CQL.FALSE) {
      return;
    }

    boolean hasAccess =
        QueryAuthorizationCheckHelper.checkAccessWithSelect(context, keyFilter, authFilter);
    if (!hasAccess) {
      // To be 100% safe against information-disclosure, we'd need to check if the
      // user can read the entity and only then return 403.
      throw new ErrorStatusException(
          CdsErrorStatuses.EVENT_FORBIDDEN, eventName, targetEntity.getQualifiedName());
    }
  }

  private boolean isRelevantEventForRejectionHandler(EventContext context) {
    String eventName = context.getEvent();
    if (eventName.equals(CqnService.EVENT_DELETE)
        || eventName.equals(CqnService.EVENT_UPDATE)
        || eventName.equals(DraftService.EVENT_DRAFT_EDIT)
        || (regardCustomEvents && !CdsModelUtils.isStandardCdsEvent(eventName))) {

      // Require a target entity, as without one, we can't check for authorization.
      // Also, only filterable CQNs are relevant for this handler, as we wouldn't
      // be able to extract any keys otherwise.
      return (context.getTarget() != null)
          && (context.get("cqn") instanceof CqnFilterableStatement);
    }
    return false;
  }
}
