/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_RANGE;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.CdsDataProcessor.Validator;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.cqn.CqnLiteral;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.ValidatorErrorUtils;
import com.sap.cds.services.impl.utils.ValidatorExecutor;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsTypeUtils;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

@ServiceName(value = "*", type = ApplicationService.class)
public class RangeAssertionHandler implements EventHandler {

  private static final String INF = "\u221e";

  @Before
  @HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
  public void runCheck(EventContext context) {
    CdsDataProcessor cdsDataProcessor =
        CdsDataProcessor.create()
            .addValidator(RangeAssertionHandler::hasAssertRange, rangeValidator(context));
    ValidatorExecutor.processEntities(cdsDataProcessor, context);
  }

  @SuppressWarnings("unchecked")
  private Validator rangeValidator(EventContext context) {
    return (path, element, value) -> {
      if (value != null) {
        Object rangeValues = ASSERT_RANGE.getListOrValue(element, null);
        Boundaries checks =
            getBoundaries(
                context, (List<Object>) rangeValues, element, element.getDeclaringType().getName());

        Comparable<Object> elementValue = (Comparable<Object>) value;

        if (!(checks.min.test.test(elementValue) && checks.max.test.test(elementValue))) {
          String range =
              checks.min.bracket
                  + checks.min.val.toString()
                  + ", "
                  + checks.max.val.toString()
                  + checks.max.bracket;
          ValidatorErrorUtils.handleValidationError(
              context,
              path,
              element,
              CdsAnnotations.ASSERT_RANGE_MESSAGE,
              CdsErrorStatuses.VALUE_OUT_OF_RANGE,
              element.getName(),
              range);
        }
      }
    };
  }

  private static boolean hasAssertRange(Path path, CdsElement element, CdsType type) {
    Object rangeValues = ASSERT_RANGE.getListOrValue(element, null);
    return rangeValues instanceof List l && l.size() == 2 && isAllowedType(type);
  }

  private static Boundaries getBoundaries(
      EventContext context, List<Object> rangeValues, CdsElement element, String entityName) {
    try {
      CdsBaseType type = element.getType().as(CdsSimpleType.class).getType();

      Check min = checkMin(type, rangeValues.get(0));
      Check max = checkMax(type, rangeValues.get(1));

      return new Boundaries(min, max);
    } catch (CdsDataException e) {
      // Throw an exception if the annotation values do not match the element
      throw new ErrorStatusException(
          CdsErrorStatuses.INVALID_ANNOTATION,
          ASSERT_RANGE.toString(),
          element.getName(),
          entityName,
          e);
    }
  }

  private static Check checkMin(CdsBaseType type, Object o) {
    Kind kind = kind(o);
    if (kind == Kind.UNLIMITED) {
      return new Check("-" + INF, v -> true, "(");
    }
    Comparable<Object> minVal = parse(type, o);
    if (kind == Kind.INCLUDING) {
      return new Check(minVal, v -> minVal.compareTo(v) <= 0, "[");
    }

    // EXCLUDING
    return new Check(minVal, v -> minVal.compareTo(v) < 0, "(");
  }

  private static Check checkMax(CdsBaseType type, Object o) {
    Kind kind = kind(o);
    if (kind == Kind.UNLIMITED) {
      return new Check(INF, v -> true, ")");
    }
    Comparable<Object> maxVal = parse(type, o);
    if (kind == Kind.INCLUDING) {
      return new Check(maxVal, v -> maxVal.compareTo(v) >= 0, "]");
    }

    // EXCLUDING
    return new Check(maxVal, v -> maxVal.compareTo(v) > 0, ")");
  }

  private enum Kind {
    INCLUDING,
    EXCLUDING,
    UNLIMITED
  }

  private static Kind kind(Object v) {
    if (v instanceof CqnLiteral<?>) { // [(5), (7)]
      return Kind.EXCLUDING;
    }
    if (v instanceof Map<?, ?> m && "_".equals(m.get("="))) { // [_, 7]
      return Kind.UNLIMITED;
    }
    return Kind.INCLUDING; // e.g. [5, 7]
  }

  @SuppressWarnings("unchecked")
  private static Comparable<Object> parse(CdsBaseType type, Object v) {
    if (v instanceof CqnLiteral l) {
      v = l.value();
    }
    return (Comparable<Object>) CdsTypeUtils.parse(type, v.toString());
  }

  @SuppressWarnings("deprecation")
  private static boolean isAllowedType(CdsType elementType) {
    if (elementType instanceof CdsSimpleType st) {
      return switch (st.getType()) {
        case INT32,
            INTEGER,
            INT64,
            INTEGER64,
            DECIMAL,
            DECIMAL_FLOAT,
            DOUBLE,
            DATE,
            TIME,
            DATETIME,
            TIMESTAMP ->
            true;
        default -> false;
      };
    }

    return false;
  }

  record Check(Object val, Predicate<Comparable<Object>> test, String bracket) {}

  record Boundaries(Check min, Check max) {}
}
