/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.cds;

import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.authorization.AuthorizationService;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.changeset.ChangeSetContextSPI;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.authorization.AuthorizationEntity;
import com.sap.cds.services.impl.authorization.StaticAuthorizationsDeepChecker;
import com.sap.cds.services.impl.draft.ParentEntityLookup;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.TenantAwareCache;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ServiceName(value = "*", type = ApplicationService.class)
public class StaticAuthorizationHandler implements EventHandler {

  private static final Logger logger = LoggerFactory.getLogger(StaticAuthorizationHandler.class);

  private final TenantAwareCache<ParentEntityLookup, CdsModel> parentEntityLookups;

  private final boolean authorizeDeeply;

  private final AuthorizationService authService;

  public StaticAuthorizationHandler(CdsRuntime runtime) {
    parentEntityLookups =
        TenantAwareCache.create(
            () -> new ParentEntityLookup(RequestContext.getCurrent(runtime).getModel()), runtime);
    authorizeDeeply =
        runtime
            .getEnvironment()
            .getCdsProperties()
            .getSecurity()
            .getAuthorization()
            .getDeep()
            .isEnabled();
    authService =
        runtime
            .getServiceCatalog()
            .getService(AuthorizationService.class, AuthorizationService.DEFAULT_NAME);
  }

  // Restriction annotations from underlying projected entities are added to the
  // service entity by the compiler provided that no restrictions are defined already.
  // So the authorization has to be tested only on service level.
  @Before(event = "*")
  @HandlerOrder(OrderConstants.Before.CHECK_STATIC_AUTHORIZATION)
  private void checkStaticAuthorization(EventContext context) {
    // privileged users just skip authorization checks
    if (!context.getUserInfo().isPrivileged()) {
      checkStaticAuthorizationOrThrow(context);
    }
  }

  private void checkStaticAuthorizationOrThrow(EventContext context) {

    final String event = context.getEvent();
    final String serviceName =
        ((ApplicationService) context.getService()).getDefinition().getQualifiedName();

    // Step #1: Check static service access for all events
    if (!authService.hasServiceAccess(serviceName, event)) {
      throwAuthorizationError(context, event, serviceName);
    }

    // Step #2: Check static entity access for all events if target entity is set.
    // Note that this might be also true for bound actions and functions
    CdsEntity targetEntity = context.getTarget();
    String entityName = null;
    if (targetEntity != null) {
      entityName = targetEntity.getQualifiedName();

      CqnStatement query = null;
      Object cqn = context.get("cqn"); // TODO better work with new interface like CdsEventContext
      if (cqn instanceof CqnStatement statement) {
        query = statement;
      }

      // Phase #1: Determine what actual entity will be used for auth check
      AuthorizationEntity authEntity =
          AuthorizationEntity.resolve(
              context, entity -> parentEntityLookups.findOrCreate().lookupParent(entity));

      if (authEntity.getAuthorizationEntity() == null) {
        // no "authorization entity" found -> reject as direct access to auto-exposed entity not
        // allowed
        logger.debug(
            "No authorization entity found when sending event '{}' to entity '{}'",
            event,
            entityName);
        throwAuthorizationError(context, event, entityName);
      }

      final String authorizationEntityName = authEntity.getAuthorizationEntity().getQualifiedName();

      // Phase #2: Check static authorizations
      if (!authService.hasEntityAccess(authorizationEntityName, event, query)) {
        throwAuthorizationError(context, event, entityName);
      }

      // Phase #2.5: Check deep authorizations
      if (authorizeDeeply && query != null) {
        // The entityName (actual target of the statement)
        // and authorizationEntityName (entity for which the initial authorization checks are
        // performed)
        // are always excluded from the deep check
        Collection<String> alreadyChecked = List.of(entityName, authorizationEntityName);

        // The deep authorization checks triggered for associations always use READ as an event.
        CqnVisitor visitor = newDeepAuthorizationChecker(context, query, alreadyChecked);
        if (query.isSelect()) {
          query.asSelect().accept(visitor);

        } else if (query.isInsert()) {
          query.asInsert().ref().accept(visitor);

        } else if (query.isUpsert()) {
          query.asUpsert().ref().accept(visitor);

        } else if (query.isUpdate()) {
          CqnUpdate update = query.asUpdate();
          update.ref().accept(visitor);
          update.where().ifPresent(w -> w.accept(visitor));

        } else if (query.isDelete()) {
          CqnDelete delete = query.asDelete();
          delete.ref().accept(visitor);
          delete.where().ifPresent(w -> w.accept(visitor));
        }
      }

      // Phase #4: Test filter of authorization entity with READ if required
      if (!authorizationEntityCondition(context, entityName, authEntity)) {
        throwAuthorizationError(context, event, entityName);
      }
    }

    String actionOrFunctionName = (targetEntity == null ? (serviceName + "." + event) : event);

    // Step #3: Check actions resp. functions
    // Note: 'event' is the function/action name and depending on entityName are bound or unbound.
    // Note: isStandardCdsEvent() is false for actions and functions
    if (!CdsModelUtils.isStandardCdsEvent(event)) {
      if (CdsModelUtils.getAction(context.getModel(), entityName, actionOrFunctionName).isPresent()
          && !authService.hasActionAccess(entityName, actionOrFunctionName)) {
        throwAuthorizationError(
            context, actionOrFunctionName, entityName != null ? entityName : serviceName);

      } else if (CdsModelUtils.getFunction(context.getModel(), entityName, actionOrFunctionName)
              .isPresent()
          && !authService.hasFunctionAccess(entityName, actionOrFunctionName)) {
        throwAuthorizationError(
            context, actionOrFunctionName, entityName != null ? entityName : serviceName);
      }
    }
    // all checks derived from the model passes here.
    logger.debug(
        "Static authorization check passed for event '{}' on service '{}'", event, serviceName);
  }

  private CqnVisitor newDeepAuthorizationChecker(
      EventContext context, CqnStatement statement, Collection<String> alreadyChecked) {
    return new StaticAuthorizationsDeepChecker(
        context,
        statement,
        alreadyChecked,
        entity -> {
          String entityName = entity.getQualifiedName();
          if (!authService.hasEntityAccess(entityName, CqnService.EVENT_READ)) {
            logger.debug(
                "Deep Authorization: user is not allowed to access entity '{}'", entityName);
            throwAuthorizationError(context, CqnService.EVENT_READ, entityName);
          }
        });
  }

  private static void throwAuthorizationError(
      EventContext context, String event, String elementName) {
    if (context.getUserInfo().isAuthenticated()) {
      throw new ErrorStatusException(CdsErrorStatuses.EVENT_FORBIDDEN, event, elementName);
    } else {
      // in case the user is not even authenticated and the authorization has failed we throw 401
      throw new ErrorStatusException(CdsErrorStatuses.EVENT_UNAUTHENTICATED, event, elementName);
    }
  }

  private boolean authorizationEntityCondition(
      EventContext context, String entityName, AuthorizationEntity authEntity) {
    // Check instance-based condition on authorization entity if required
    // We can't test by sending READ on the service level as 'event' implies a different conditions
    // in general.
    // Instead, we take the condition of 'event' and test the visibility of the instance with the
    // condition applied.
    // Note on drafts: the authorization is enforced by DraftHandler which will filter all draft
    // documents
    // with the correct owner.
    if (authEntity.getAuthorizationPath() != null
        && !entityName.equals(authEntity.getAuthorizationEntity().getQualifiedName())
        && authEntity.authorizationCondition(context.getEvent()) != null
        && !authEntity.isDraft()) {

      @SuppressWarnings("resource")
      ChangeSetContextSPI changeSetContext = (ChangeSetContextSPI) context.getChangeSetContext();
      // TODO adapt this, if authorization handler starts to support multiple persistence services
      if (changeSetContext == null
          || !changeSetContext.hasChangeSetMember(PersistenceService.DEFAULT_NAME)) {
        return context
            .getCdsRuntime()
            .changeSetContext()
            .run(
                (Function<ChangeSetContext, Boolean>)
                    c -> authorizationEntityConditionImpl(context, authEntity));
      } else {
        return authorizationEntityConditionImpl(context, authEntity);
      }
    }
    return true;
  }

  private boolean authorizationEntityConditionImpl(
      EventContext context, AuthorizationEntity authEntity) {
    CqnPredicate authorizationCondition = authEntity.authorizationCondition(context.getEvent());
    if (authorizationCondition != null) {
      CqnSelect readTest =
          Select.from(authEntity.getAuthorizationPath()).where(authorizationCondition);
      // TODO: This makes generic authorization handler dependent from persistence
      // service, but sending to the corresponding application server is not correct (generic and
      // custom handlers
      // would be called with wrong event 'READ').
      // We decided to tackle this general problem when we start to support several persistence
      // service instances.
      if (CdsServiceUtils.getDefaultPersistenceService(context).run(readTest).first().isEmpty()) {
        logger.debug(
            "No authorization to send event '{}' to entity '{}' because of instance '{}'",
            context.getEvent(),
            authEntity.getAuthorizationEntity().getQualifiedName(),
            authEntity.getAuthorizationPath());
        return false;
      }
    }
    return true;
  }
}
