/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.draft;

import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Cached lookups for parent entities. */
public class ParentEntityLookup {

  private static final Logger log = LoggerFactory.getLogger(ParentEntityLookup.class);

  private Map<CdsEntity, List<ParentEntityLookupResult>> cachedParentEntities =
      new ConcurrentHashMap<>();

  private WeakReference<CdsModel> model;

  public ParentEntityLookup(CdsModel model) {
    // if access to lookup is given via TenantAwareCache, existance of model is also given
    this.model = new WeakReference<>(model);
  }

  public List<ParentEntityLookupResult> lookupParent(CdsEntity child) {
    if (!cachedParentEntities.containsKey(child)) {
      List<ParentEntityLookupResult> parentEntities = new ArrayList<>();
      model
          .get()
          .entities()
          .forEach(
              entity -> {
                Stream<CdsElement> compositions =
                    entity
                        .compositions()
                        .filter(c -> c.getType().as(CdsAssociationType.class).getTarget() == child);
                compositions.forEach(
                    composition -> {
                      ParentEntityLookupResult result =
                          new ParentEntityLookupResult(entity, composition);
                      log.debug("Found {} via {} as parent for {}", entity, composition, child);
                      parentEntities.add(result);
                    });
              });
      cachedParentEntities.putIfAbsent(child, parentEntities);
    }
    return cachedParentEntities.get(child);
  }

  public static class ParentEntityLookupResult {
    private CdsEntity parentEntity;
    private CdsElement composition;

    public ParentEntityLookupResult(CdsEntity parentEntity, CdsElement composition) {
      this.parentEntity = parentEntity;
      this.composition = composition;
    }

    public CdsEntity getParentEntity() {
      return parentEntity;
    }

    public CdsElement getComposition() {
      return composition;
    }
  }
}
