/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.draft.messages;

import static com.sap.cds.services.impl.draft.messages.DraftMessageUtils.MESSAGES_ELEMENT;

import com.sap.cds.ql.CQL;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.changeset.ChangeSetListener;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.utils.OrderConstants;
import java.util.List;

@ServiceName(value = "*", type = DraftService.class)
class DraftMessagesActionHandler implements EventHandler {

  @On
  @HandlerOrder(OrderConstants.On.FEATURE)
  void draftActivate(DraftSaveEventContext context) {
    ServiceException exception = null;
    try {
      context.proceed();
    } catch (ServiceException e) {
      exception = e;
      throw e;
    } finally {
      List<DraftMessage> draftMessages =
          DraftMessageUtils.toDraftMessages(exception, context.getMessages());
      context
          .getChangeSetContext()
          .register(
              new ChangeSetListener() {

                @Override
                public void afterClose(boolean completed) {
                  if (!completed) { // messages are stored with the draft, which doesn't exist
                    // anymore if the activation succeeded
                    context
                        .getCdsRuntime()
                        .requestContext()
                        .clearMessages()
                        .run(
                            requestContext -> {
                              DraftMessagesWriteHandler.persist(
                                  context.getCqn().ref(),
                                  context.getCqn().where().orElse(CQL.TRUE),
                                  draftMessages,
                                  context);
                            });
                  }
                }
              });
    }
  }

  @On
  @HandlerOrder(OrderConstants.On.FEATURE)
  void draftPrepare(DraftPrepareEventContext context) {
    String select = context.getParameterInfo().getQueryParameter("$select");
    if (select == null || !select.contains(MESSAGES_ELEMENT)) {
      return; // no messages requested
    }

    // simulate validation
    context
        .getCdsRuntime()
        .requestContext()
        .clearMessages()
        .run(
            requestContext -> {
              context
                  .getCdsRuntime()
                  .changeSetContext()
                  .run(
                      changeSetContext -> {
                        try {
                          context.getService().saveDraft(context.getCqn());
                        } catch (ServiceException e) {
                          // ignore
                        } finally {
                          changeSetContext.markForCancel();
                        }
                      });
            });
  }
}
