/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.handlerregistry;

import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.handlerregistry.resolver.ArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.CqnReferenceArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.CqnStatementArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.EventContextArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.ObjectReturnResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.PojoArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.ResultReturnResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.ReturnResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.ServiceArgumentResolver;
import com.sap.cds.services.impl.handlerregistry.resolver.VoidReturnResolver;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * Helper class to iterate through all methods of a given class and calculate the {@link
 * HandlerDescriptor} objects for all event handler methods
 */
public class AnnotatedHandlerMethodsFinder {

  private static final Lookup lookup = MethodHandles.lookup();

  private final Map<String, HandlerDescriptor> handlerDescriptors = new HashMap<>();
  private ServiceName closestDefaultServiceName = null;

  /**
   * Processes a class and creates {@link HandlerDescriptor} objects for each event handler method
   *
   * @param clazz the handler class to be processed
   */
  public AnnotatedHandlerMethodsFinder(Class<?> clazz) {
    iterate(clazz);

    for (HandlerDescriptor descriptor : getHandlerDescriptors()) {
      if (closestDefaultServiceName != null) {
        descriptor.setDefaultServiceNames(closestDefaultServiceName.value());
        descriptor.setDefaultServiceTypes(closestDefaultServiceName.type());
      }
      descriptor.verifyOrThrow();
    }
  }

  /**
   * @return all collected {@link HandlerDescriptor} objects
   */
  public Collection<HandlerDescriptor> getHandlerDescriptors() {
    return handlerDescriptors.values();
  }

  /**
   * Iterates over the complete class hierarchy of a class, including interfaces. Each class in the
   * hierarchy is processed and searched for event handler annotations.
   *
   * @param cls the handler class to be processed
   */
  private void iterate(Class<?> cls) {
    if (cls != null && cls != Object.class) {

      iterate(cls.getSuperclass());

      for (Class<?> interf : cls.getInterfaces()) {
        iterate(interf);
      }

      processClass(cls);
    }
  }

  /**
   * Processes a single class and checks all its methods for event handler annotations. It takes
   * care of creating a {@link HandlerDescriptor} once any event handler annotations are observed on
   * a method signature.
   *
   * @param cls the class to be processed
   */
  private void processClass(Class<?> cls) {
    for (Method method : cls.getDeclaredMethods()) {
      String methodKey = toMethodKey(method);
      String methodName = method.toString();
      HandlerDescriptor descriptor = handlerDescriptors.get(methodKey);

      // check if a phase annotation can be found
      Before eBefore = method.getAnnotation(Before.class);
      On eOn = method.getAnnotation(On.class);
      After eAfter = method.getAnnotation(After.class);
      if (!(eBefore == null && eOn == null && eAfter == null)) {
        // create the descriptor, if it does not yet exist
        // the descriptor is created on the first occurrence of a phase annotation
        if (descriptor == null) {
          MethodHandle methodHandle;

          try {
            method.setAccessible(true);
            methodHandle = lookup.unreflect(method);
            // allow passing an Object[], which will be spread
            // the first parameter is the object instance on which the method is called, as defined
            // by unreflect()
            methodHandle = methodHandle.asSpreader(Object[].class, method.getParameterCount() + 1);
          } catch (IllegalAccessException e) {
            throw new ErrorStatusException(
                CdsErrorStatuses.HANDLER_NOT_ACCESSIBLE, method.getName(), e);
          }

          descriptor = new HandlerDescriptor(methodName, methodHandle, method.getParameterCount());

          for (Parameter parameter : method.getParameters()) {
            ArgumentResolver argumentResolver =
                EventContextArgumentResolver.createIfApplicable(parameter.getType());

            if (argumentResolver == null) {
              argumentResolver =
                  CqnStatementArgumentResolver.createIfApplicable(parameter.getType());
            }

            if (argumentResolver == null) {
              argumentResolver =
                  CqnReferenceArgumentResolver.createIfApplicable(parameter.getType());
            }

            if (argumentResolver == null) {
              argumentResolver =
                  PojoArgumentResolver.createIfApplicable(
                      parameter.getType(), parameter.getParameterizedType());
            }

            if (argumentResolver == null) {
              argumentResolver = ServiceArgumentResolver.createIfApplicable(parameter.getType());
            }

            if (argumentResolver == null) {
              throw new ErrorStatusException(
                  CdsErrorStatuses.RESOLVING_PARAMETER_TYPE_FAILED,
                  parameter.getType().getName(),
                  methodName);
            }

            descriptor.getArgumentResolvers().add(argumentResolver);
          }

          Class<?> returnType = method.getReturnType();
          ReturnResolver returnResolver = VoidReturnResolver.createIfApplicable(returnType);

          if (returnResolver == null) {
            returnResolver = ResultReturnResolver.createIfApplicable(method.getGenericReturnType());
          }

          if (returnResolver == null) {
            returnResolver = ObjectReturnResolver.createIfApplicable();
          }

          descriptor.setReturnResolver(returnResolver);

          handlerDescriptors.put(methodKey, descriptor);
        }

        // last found declaration of these wins
        if (eBefore != null) {
          descriptor.setBefore(eBefore);
        }

        if (eOn != null) {
          descriptor.setOn(eOn);
        }

        if (eAfter != null) {
          descriptor.setAfter(eAfter);
        }
      }

      // these can only be set, after at least one phase annotation has been processed
      if (descriptor != null) {
        // last found declaration of these wins
        HandlerOrder order = method.getAnnotation(HandlerOrder.class);
        if (order != null) {
          descriptor.setOrder(order.value());
        }
      }
    }

    // last found declaration of these wins
    ServiceName defaultServiceName = cls.getAnnotation(ServiceName.class);
    if (defaultServiceName != null) {
      closestDefaultServiceName = defaultServiceName;
    }
  }

  /**
   * Returns a key for this method, which is guaranteed to not differ between the methods definition
   * in interfaces, abstract classes and classes. It is only based on the method name (without its
   * defining class) and the parameter types
   *
   * @param m the method
   * @return the key string
   */
  private String toMethodKey(Method m) {
    StringBuilder sb = new StringBuilder(256);
    sb.append(m.getName());
    sb.append('(');
    Class<?>[] p = m.getParameterTypes();
    for (int i = 0; i < p.length; ++i) {
      if (i != 0) {
        sb.append(',');
      }
      sb.append(p[i].getName());
    }
    sb.append(')');
    return sb.toString();
  }
}
