/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.outbox;

import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.outbox.OutboxMessage;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.outbox.OutboxUtils;
import java.time.Instant;

/**
 * Abstract implementation of {@link OutboxService}, providing base implementation for all specific
 * outbox services.
 */
public abstract class AbstractOutboxService extends ServiceDelegator implements OutboxService {

  protected final CdsRuntime runtime;

  protected AbstractOutboxService(String name, CdsRuntime runtime) {
    super(name);
    this.runtime = runtime;
  }

  @Override
  public void submit(String outboxEvent, OutboxMessage message) {
    OutboxMessageEventContext context = OutboxMessageEventContext.create(outboxEvent);
    context.setMessage(message);
    emit(context);
  }

  @Override
  public <S extends Service> S outboxed(S service) {
    return OutboxedServiceProxyUtils.outboxed(this, service, runtime);
  }

  @Override
  public <A extends Service> A outboxed(Service service, Class<A> asyncInterface) {
    return OutboxedServiceProxyUtils.outboxed(this, service, asyncInterface, runtime);
  }

  @On
  @HandlerOrder(OrderConstants.On.PRIORITY)
  private void storeOutboxMessage(OutboxMessageEventContext context) {
    if (Boolean.FALSE.equals(context.getIsInbound())) {

      if (context.getTimestamp() == null) {
        context.setTimestamp(Instant.now());
      }

      // store request context
      RequestContext requestContext = RequestContext.getCurrent(context.getCdsRuntime());
      OutboxUtils.storeRequestContext(requestContext, context.getMessage());

      submit(context);
      context.setCompleted();
    }
  }

  @On
  @HandlerOrder(OrderConstants.On.DEFAULT_ON)
  private void publishOutboxMessage(OutboxMessageEventContext context) {
    final Service service = context.getServiceCatalog().getService(context.getEvent());
    if (service == null) {
      return;
    }

    OutboxService.unboxed(service).emit(OutboxUtils.toEventContext(context));
    context.setCompleted();
  }

  /**
   * Stores the outbox message in the outbox, to be retrieved for later publishing.
   *
   * @param context the {@link OutboxMessageEventContext}
   */
  protected abstract void submit(OutboxMessageEventContext context);
}
