/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.outbox;

import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.changeset.ChangeSetListener;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.outbox.OutboxUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryOutbox extends AbstractOutboxService {

  private static final Logger logger = LoggerFactory.getLogger(InMemoryOutbox.class);

  public InMemoryOutbox(String name, CdsRuntime runtime) {
    super(name, runtime);
  }

  @Override
  protected void submit(OutboxMessageEventContext context) {
    logger.debug("Submitting outbox message with outbox event '{}'", context.getEvent());
    boolean emitDuringChangeSetContext =
        context
            .getCdsRuntime()
            .getEnvironment()
            .getCdsProperties()
            .getOutbox()
            .getInMemory()
            .isEmitDuringChangeSetContext();

    ChangeSetContext changeSet = context.getChangeSetContext();

    changeSet.register(
        new ChangeSetListener() {
          @Override
          public void beforeClose() {
            if (!changeSet.isMarkedForCancel() && emitDuringChangeSetContext) {
              publishMessage(context);
            }
          }

          @Override
          public void afterClose(boolean completed) {
            if (!emitDuringChangeSetContext && completed) {
              publishMessage(context);
            }
          }
        });
  }

  private void publishMessage(OutboxMessageEventContext submitContext) throws ErrorStatusException {
    // restore request context
    RequestContextRunner runner = this.runtime.requestContext();
    OutboxUtils.restoreRequestContext(runner, submitContext.getMessage());
    runner.run(
        requestContext -> {
          logger.debug(
              "Publishing outbox message with outbox event '{}'", submitContext.getEvent());
          OutboxMessageEventContext publishContext =
              OutboxMessageEventContext.create(submitContext.getEvent());
          publishContext.setIsInbound(true);
          publishContext.setTimestamp(submitContext.getTimestamp());
          publishContext.setMessage(submitContext.getMessage());
          submitContext.getService().emit(publishContext);
        });
  }
}
