/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.outbox;

import static com.sap.cds.services.utils.CdsErrorStatuses.SERVICE_HAS_NO_SERVICESPI;

import com.sap.cds.services.Service;
import com.sap.cds.services.impl.ServiceSPI;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.ErrorStatusException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

public class OutboxedServiceInvocationHandler implements InvocationHandler {

  private final OutboxService outboxService;
  private final Service service;
  private final ServiceSPI serviceSPI;
  private final CdsRuntime runtime;

  public OutboxedServiceInvocationHandler(
      OutboxService outboxService, Service service, CdsRuntime runtime) {
    this.outboxService = outboxService;
    this.service = service;
    this.serviceSPI = CdsServiceUtils.getServiceSPI(service);
    this.runtime = runtime;

    if (this.serviceSPI == null) {
      throw new ErrorStatusException(SERVICE_HAS_NO_SERVICESPI, service.getName());
    }
  }

  public OutboxService getOutboxService() {
    return outboxService;
  }

  public Service getDelegatedService() {
    return service;
  }

  @Override
  public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
    OutboxService targetOutbox = outboxService;
    // events of the provider tenant can't be stored, if MT is enabled, as there is no persistence
    // for the provider.
    boolean providerTenantPersistence =
        runtime.getEnvironment().getCdsProperties().getOutbox().getProviderTenant().isEnabled();
    if (!(outboxService instanceof InMemoryOutbox)
        && !providerTenantPersistence
        && RequestContext.getCurrent(runtime).getUserInfo().getTenant() == null) {
      targetOutbox =
          runtime
              .getServiceCatalog()
              .getService(InMemoryOutbox.class, OutboxService.INMEMORY_NAME); // could be null
    }

    try {
      serviceSPI.setOutbox(targetOutbox);
      return method.invoke(service, args);
    } finally {
      serviceSPI.setOutbox(null);
    }
  }
}
