/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.scheduler;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.runtime.CdsRuntime;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskScheduler implements Runnable {

  private static final Logger LOG = LoggerFactory.getLogger(TaskScheduler.class);

  private final Map<String, TaskInfo> tasksTable = new ConcurrentHashMap<>();
  private final Object controllerMonitor = new Object();
  private final Thread controllerThread = new Thread(this, "task-scheduler-controller");

  private final int poolSize;
  private final Duration lookupInterval;
  private volatile boolean doPause = true;
  private ThreadPoolExecutor executor;
  private SchedulerListener listener = new SchedulerListener();

  @VisibleForTesting
  TaskScheduler(int poolSize, long lookupIntervalMillis) {
    this.poolSize = poolSize;
    this.lookupInterval = Duration.ofMillis(lookupIntervalMillis);
    start(); // start immediately for tests
  }

  public TaskScheduler(CdsRuntime runtime) {
    CdsProperties config = runtime.getEnvironment().getCdsProperties();
    this.poolSize = config.getTaskScheduler().getThreadPoolSize();
    this.lookupInterval = config.getTaskScheduler().getLookupInterval();
  }

  public void start() {
    this.executor =
        new ThreadPoolExecutor(
            poolSize,
            poolSize,
            1,
            TimeUnit.MINUTES,
            new LinkedBlockingQueue<>(),
            new ThreadFactoryBuilder().setNameFormat("task-scheduler-%d").setDaemon(true).build());
    this.controllerThread.setDaemon(true);
    this.controllerThread.start();
    LOG.info(
        "Started Task Scheduler with pool size '{}' and lookup interval '{}'",
        poolSize,
        lookupInterval);
  }

  public void shutdown() {
    controllerThread.interrupt();
    if (executor != null) {
      executor.shutdownNow();
    }
  }

  public void scheduleTask(String key, Task task, long inMillis) {
    if (executor == null || executor.isShutdown()) {
      return;
    }

    listener.scheduledNewTask(
        tasksTable.compute(
            key,
            (k, taskInfo) -> {
              if (taskInfo == null) {
                LOG.debug("Scheduling new task '{}' in {} millis", key, inMillis);
                return new TaskInfo(key, System.currentTimeMillis() + inMillis, task);
              }
              // recalculate schedule time in case task was already scheduled
              if (taskInfo.setScheduleTsIfLower(System.currentTimeMillis() + inMillis)) {
                LOG.debug("Reduced wait time for existing task '{}' to {} millis", key, inMillis);
              }
              return taskInfo;
            }));

    // wake up lookup thread to submit the task
    wakeupScheduler();
  }

  /** Wakes up the scheduler thread if its sleeping. */
  private void wakeupScheduler() {
    if (doPause) {
      synchronized (controllerMonitor) {
        if (doPause) {
          controllerMonitor.notifyAll();
          doPause = false;
        }
      }
    }
  }

  /** Periodic or wake up lookup of tasks */
  @Override
  public void run() {
    long lastMinInterval = lookupInterval.toMillis();
    while (await(lastMinInterval)) {
      long currentTs = System.currentTimeMillis();
      lastMinInterval = lookupInterval.toMillis();
      for (TaskInfo taskInfo : tasksTable.values()) {
        if (!taskInfo.submitted && taskInfo.task.isReady()) {
          // avoid concurrent modification of local value
          long scheduleTs = taskInfo.scheduleTs;
          if (scheduleTs <= currentTs) {
            try {
              submitTask(taskInfo);
            } catch (RejectedExecutionException e) {
              break; // shutdown
            }
          } else {
            lastMinInterval = Math.min(lastMinInterval, scheduleTs - currentTs);
          }
        }
      }
    }
  }

  /**
   * Awaits a scheduler wakeup or a the passing of the sleep time.
   *
   * @return false, if the controllerThread should be terminated
   */
  private boolean await(long pauseTime) {
    synchronized (controllerMonitor) {
      try {
        if (doPause) {
          listener.sleeping(pauseTime);
          controllerMonitor.wait(pauseTime);
        }
        doPause = true;
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
        return false;
      }
    }
    return !executor.isShutdown();
  }

  private void submitTask(TaskInfo taskInfo) {
    LOG.debug("Submitting task '{}'", taskInfo.key);
    taskInfo.submitted = true;
    executor.submit(
        () -> {
          TaskSchedule result = null;
          try {
            LOG.debug("Starting task '{}'", taskInfo.key);
            listener.taskStarted(taskInfo);
            result = taskInfo.task.run();
          } finally {
            LOG.debug("Task '{}' finished", taskInfo.key);
            if (result == null) {
              tasksTable.remove(taskInfo.key);
            } else {
              LOG.debug("Rescheduling task '{}' in {} millis", taskInfo.key, result.delay);
              taskInfo.setScheduleTs(
                  System.currentTimeMillis() + result.delay, result.allowEarlier);
              taskInfo.task = result.task;
              taskInfo.submitted = false;
              wakeupScheduler();
            }
            listener.taskFinished(taskInfo);
          }
        });
    listener.submittedNewTask(taskInfo);
  }

  /** Task meta data used for enqueuing */
  public static class TaskInfo {

    private final String key;
    private volatile long scheduleTs;
    private volatile boolean allowEarlier = true;
    private volatile boolean submitted;
    @JsonIgnore private volatile Task task;

    TaskInfo(String key, long scheduleTs, Task task) {
      this.key = key;
      this.task = task;
      this.scheduleTs = scheduleTs;
    }

    private synchronized void setScheduleTs(long newScheduleTs, boolean allowEarlier) {
      this.allowEarlier = allowEarlier;
      this.scheduleTs = newScheduleTs;
    }

    private boolean setScheduleTsIfLower(long newScheduleTs) {
      if (newScheduleTs < scheduleTs) {
        synchronized (this) {
          if (allowEarlier && newScheduleTs < scheduleTs) {
            scheduleTs = newScheduleTs;
            return true;
          }
        }
      }
      return false;
    }

    @JsonGetter("suspended")
    public boolean isSuspended() {
      return !task.isReady();
    }

    @JsonGetter("key")
    public String getKey() {
      return key;
    }

    @JsonGetter("scheduleTs")
    public long getScheduleTs() {
      return scheduleTs;
    }

    @JsonGetter("submitted")
    public boolean getSubmmited() {
      return submitted;
    }
  }

  public interface Task {
    /**
     * Executes the task
     *
     * @return a {@link TaskSchedule} to describe the next task that should be scheduled or {@code
     *     null}, if no further tasks should be scheduled.
     */
    TaskSchedule run();

    /**
     * @return true, if the task is ready for execution.
     */
    default boolean isReady() {
      return true;
    }
  }

  public static record TaskSchedule(Task task, long delay, boolean allowEarlier) {

    public TaskSchedule(Task task) {
      this(task, 0, false);
    }
  }
  ;

  // For Monitoring Only

  public void setListener(SchedulerListener listener) {
    this.listener = listener;
  }

  public List<TaskInfo> getTasksSchedule() {
    return tasksTable.values().stream()
        .sorted((o1, o2) -> (int) (o1.scheduleTs - o2.scheduleTs))
        .toList();
  }

  public static class SchedulerListener {
    protected void scheduledNewTask(TaskInfo info) {}

    public void submittedNewTask(TaskInfo taskInfo) {}

    public void sleeping(long pauseTime) {}

    protected void taskFinished(TaskInfo taskInfo) {}

    protected void taskStarted(TaskInfo taskInfo) {}
  }
}
