/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.utils;

import com.sap.cds.Result;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnXsert;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.impl.EventContextDelegator;
import com.sap.cds.services.impl.EventContextSPI;
import com.sap.cds.services.impl.ServiceSPI;
import com.sap.cds.services.impl.cds.TypedCqnServiceInvocationHandler;
import com.sap.cds.services.impl.outbox.OutboxedServiceInvocationHandler;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.SessionContextUtils;
import com.sap.cds.util.PathExpressionResolver;
import java.lang.reflect.Proxy;
import java.util.List;
import java.util.Map;

/** Utility class for CDS Services */
public class CdsServiceUtils {

  /**
   * Returns the default persistence service.
   *
   * @param context the current context
   * @return the default persistence service
   */
  public static PersistenceService getDefaultPersistenceService(EventContext context) {
    return context
        .getServiceCatalog()
        .getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
  }

  /**
   * Returns the Service casted to the {@link ServiceSPI}
   *
   * @param service the {@link Service} to be casted
   * @return the Service casted to the {@link ServiceSPI}, or null
   */
  public static ServiceSPI getServiceSPI(Service service) {
    if (service instanceof ServiceSPI pI) {
      return pI;
    } else if (service instanceof ServiceDelegator delegator) {
      return getServiceSPI(delegator.getDelegatedService());
    } else if (service != null
        && Proxy.isProxyClass(service.getClass())
        && Proxy.getInvocationHandler(service)
            instanceof TypedCqnServiceInvocationHandler proxyHandler) {
      return getServiceSPI(proxyHandler.getDelegatedService());
    } else if (service != null
        && Proxy.isProxyClass(service.getClass())
        && Proxy.getInvocationHandler(service)
            instanceof OutboxedServiceInvocationHandler proxyHandler) {
      return getServiceSPI(proxyHandler.getDelegatedService());
    }
    return null;
  }

  /**
   * Returns the non-dynamic type of the {@link Service}, if possible. If the service is a typed
   * proxy, the generated interface is returned.
   *
   * @param service the {@link Service}
   * @return the best guess type of the {@link Service}
   */
  public static Class<?> getServiceType(Service service) {
    Class<?> clazz = service.getClass();
    if (Proxy.isProxyClass(clazz)) {
      Class<?>[] interfaces = clazz.getInterfaces();
      if (interfaces.length == 1) {
        // proxy class can be replaced by matching interface
        return interfaces[0];
      }
    }
    return clazz;
  }

  /**
   * Returns the list of entity maps present in the {@link EventContext}. It inspects the CQN of the
   * event and returns the entity data from it.
   *
   * @param context the {@link EventContext}
   * @return the list of entity maps available in the CQN of the {@link EventContext}
   */
  public static List<Map<String, Object>> getEntities(EventContext context) {
    Object cqn = context.get("cqn");
    if (cqn instanceof CqnXsert x) {
      return x.entries();
    } else if (cqn instanceof CqnUpdate u) {
      return u.entries();
    } else {
      throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EVENT, context.getEvent());
    }
  }

  /**
   * Returns the list of entity maps present in the {@link EventContext}. It inspects the CQN of the
   * events and returns the entity data from it. In contrast to {link {@link
   * #getEntities(EventContext)} the data also contains the resolved foreign keys from parent
   * entities in path expressions.
   *
   * @param context the {@link EventContext}
   * @return the list of entity maps available in the CQN of the {@link EventContext}
   */
  public static List<Map<String, Object>> getEntitiesResolved(EventContext context) {
    Object cqn = context.get("cqn");
    if (cqn instanceof CqnXsert x) {
      return PathExpressionResolver.resolvePath(
              context.getModel(), x, SessionContextUtils.toSessionContext(context))
          .entries();
    } else if (cqn instanceof CqnUpdate u) {
      return PathExpressionResolver.resolvePath(context.getModel(), u).entries();
    } else {
      throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EVENT, context.getEvent());
    }
  }

  /**
   * Returns the {@link Result} present in the {@link EventContext}.
   *
   * @param context the {@link EventContext}
   * @return the {@link Result} present in the {@link EventContext}
   */
  public static Result getResult(EventContext context) {
    if (context.get("result") instanceof Result r) {
      return r;
    } else {
      throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EVENT, context.getEvent());
    }
  }

  /**
   * Returns the EventContext casted to the {@link EventContextSPI}
   *
   * @param eventContext the {@link EventContext} to be casted
   * @return the Service casted to the {@link EventContextSPI}, or null
   */
  public static EventContextSPI getEventContextSPI(EventContext eventContext) {
    if (eventContext instanceof EventContextSPI pI) {
      return pI;
    } else if (eventContext instanceof EventContextDelegator delegator) {
      return getEventContextSPI(delegator.getDelegatedEventContext());
    }
    return null;
  }
}
