/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.utils;

import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.environment.CdsProperties.Errors;
import com.sap.cds.services.messages.LocalizedMessageProvider;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.LocaleUtils;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.ResourceBundle;
import java.util.Set;
import org.slf4j.helpers.MessageFormatter;

public class DefaultTranslationUtils {

  public static final String BUNDLE_NAME = "com.sap.cds.i18n.errors";
  private static final Set<String> KEYS =
      ResourceBundle.getBundle(BUNDLE_NAME, Locale.ROOT).keySet();
  private static final Map<String, ErrorStatus> errorStatuses = new HashMap<>();

  static {
    for (ErrorStatus status : ErrorStatuses.values()) {
      errorStatuses.put(status.getCodeString(), status);
    }
    for (ErrorStatus status : CdsErrorStatuses.values()) {
      errorStatuses.put(status.getCodeString(), status);
    }
  }

  public static Optional<String> getErrorStatusDescription(String code, Object[] args) {
    ErrorStatus status = errorStatuses.get(code);
    if (status != null) {
      return Optional.of(MessageFormatter.arrayFormat(status.getDescription(), args).getMessage());
    }
    return Optional.empty();
  }

  public static String getLocalizedMessage(
      LocalizedMessageProvider provider,
      String code,
      Object[] args,
      Locale locale,
      Errors config,
      boolean errorStatusFallback) {
    String localized = provider.get(code, args, locale);
    // fallback to default translations for maintained error codes
    if (config.getDefaultTranslations().isEnabled()
        && KEYS.contains(code)
        && Objects.equals(code, localized)) {
      Locale theLocale =
          locale == null ? Locale.getDefault() : LocaleUtils.getLocaleForBundle(locale);
      ResourceBundle bundle = ResourceBundle.getBundle(BUNDLE_NAME, theLocale);
      if (bundle.containsKey(code)) {
        return new MessageFormat(bundle.getString(code), theLocale).format(args);
      }
    }
    // fallback to default description for ErrorStatus
    if (errorStatusFallback) {
      ErrorStatus status = errorStatuses.get(code);
      if (status != null && Objects.equals(code, localized)) {
        return MessageFormatter.arrayFormat(status.getDescription(), args).getMessage();
      }
    }
    return localized; // no (localized) message available or desired
  }
}
