/*
 * © 2020-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.utils;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.function.Supplier;

public class ModifiedValues {

  private final Map<String, Object> modifiedValues = new TreeMap<>();

  public interface Property<T> {
    T getValue();

    T getModifiedValue();

    boolean isModified();

    void modifyValue(T value);
  }

  public <T> Property<T> field(String name, Supplier<T> prevValue) {
    return new PropertyImpl<T>(name, prevValue);
  }

  @SuppressWarnings("unchecked")
  class PropertyImpl<T> implements Property<T> {

    final String name;
    final Supplier<T> prevValue;

    PropertyImpl(String name, Supplier<T> prevValue) {
      this.name = name;
      this.prevValue = prevValue;
    }

    @Override
    public T getValue() {
      if (isModified()) {
        return getModifiedValue();
      }
      return prevValue.get();
    }

    @Override
    public T getModifiedValue() {
      return (T) modifiedValues.get(name);
    }

    @Override
    public boolean isModified() {
      return modifiedValues.containsKey(name);
    }

    @Override
    public void modifyValue(T value) {
      modifiedValues.put(name, value);
    }
  }

  public interface MapProperty<K, V> extends Property<Map<K, V>> {
    V getMapValue(K key);

    void modifyMapValue(K key, V value);
  }

  public <K, V> MapProperty<K, V> field(
      String name, Supplier<Map<K, V>> prevValue, Function<K, V> prevMapValue) {
    return new MapPropertyImpl<K, V>(name, prevValue, prevMapValue);
  }

  public <K, V> MapProperty<K, V> field(
      String name,
      Supplier<Map<K, V>> prevValue,
      Function<K, V> prevMapValue,
      Comparator<K> comparator) {
    return new MapPropertyImpl<K, V>(name, prevValue, prevMapValue, comparator);
  }

  class MapPropertyImpl<K, V> implements MapProperty<K, V> {

    final String name;
    final Supplier<Map<K, V>> prevValue;
    final Function<K, V> prevMapValue;
    boolean overridden; // in case previous map is not visible anymore

    MapPropertyImpl(String name, Supplier<Map<K, V>> prevValue, Function<K, V> prevMapValue) {
      this(name, prevValue, prevMapValue, null);
    }

    MapPropertyImpl(
        String name,
        Supplier<Map<K, V>> prevValue,
        Function<K, V> prevMapValue,
        Comparator<K> comparator) {
      this.name = name;
      this.prevValue = prevValue;
      this.prevMapValue = prevMapValue;
      modifiedValues.put(name, (comparator == null) ? new HashMap<>() : new TreeMap<>(comparator));
    }

    @Override
    public V getMapValue(K key) {
      Map<K, V> modified = getModifiedValue();
      if (modified != null && modified.containsKey(key)) {
        return modified.get(key);
      }
      if (overridden) {
        return null;
      } else {
        return prevMapValue.apply(key);
      }
    }

    @Override
    public void modifyMapValue(K key, V value) {
      Map<K, V> modified = getModifiedValue();
      modified.put(key, value);
    }

    @Override
    public Map<K, V> getValue() {

      Map<K, V> modifiedMapValue = getModifiedValue();
      if (overridden) {
        return modifiedMapValue;
      }

      Map<K, V> prevMapValue = prevValue.get();
      if (prevMapValue == null) {
        return modifiedMapValue;
      }

      // mix of both
      return new CombinedMap<>(modifiedMapValue, prevMapValue);
    }

    @Override
    public void modifyValue(Map<K, V> mapValue) {
      modifiedValues.put(name, mapValue);
      overridden = true;
    }

    @Override
    public boolean isModified() {
      return true;
    }

    @Override
    @SuppressWarnings("unchecked")
    public Map<K, V> getModifiedValue() {
      return (Map<K, V>) modifiedValues.get(name);
    }
  }
}
