/*
 * © 2022-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.impl.utils;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsKind;
import com.sap.cds.reflect.CdsParameter;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.CdsAnnotatableImpl.CdsAnnotationImpl;
import com.sap.cds.reflect.impl.CdsElementBuilder;
import com.sap.cds.reflect.impl.CdsStructuredTypeBuilder;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.impl.draft.messages.DraftMessageUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.tuple.Pair;

public class ValidatorExecutor {

  private static final Set<String> crudEvents =
      Set.of(CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT);
  private static final Set<String> draftValidationEvents =
      Set.of(DraftService.EVENT_DRAFT_NEW, DraftService.EVENT_DRAFT_PATCH);

  private ValidatorExecutor() {
    // empty
  }

  public static void processEntities(CdsDataProcessor processor, EventContext context) {
    runValidation(processor, context, () -> CdsServiceUtils.getEntities(context), true);
  }

  static void processResolvedEntities(CdsDataProcessor processor, EventContext context) {
    runValidation(processor, context, () -> CdsServiceUtils.getEntitiesResolved(context), false);
  }

  private static void runValidation(
      CdsDataProcessor processor,
      EventContext context,
      Supplier<List<Map<String, Object>>> entities,
      boolean includeDraftNew) {
    if (isValidationEvent(context, includeDraftNew)) {
      processor.process(entities.get(), context.getTarget());
    } else if (context.getService() instanceof ApplicationService
        && context
            .getCdsRuntime()
            .getEnvironment()
            .getCdsProperties()
            .getQuery()
            .getValidation()
            .getParameters()
            .isEnabled()
        && !CdsModelUtils.isDraftEvent(context.getEvent())) {

      handleOperationParameters(processor, context);
    }
  }

  public static boolean isValidationEvent(EventContext context, boolean includeDraftNew) {
    return crudEvents.contains(context.getEvent()) || validateDraft(context, includeDraftNew);
  }

  private static boolean validateDraft(EventContext context, boolean includeDraftNew) {
    return draftValidationEvents.contains(context.getEvent())
        && (includeDraftNew || !context.getEvent().equals(DraftService.EVENT_DRAFT_NEW))
        && DraftMessageUtils.isDraftMessagesStored(context);
  }

  private static void handleOperationParameters(CdsDataProcessor processor, EventContext context) {
    getOperationAsStructuredType(context)
        .ifPresent(definition -> processor.process(definition.getValue(), definition.getKey()));
  }

  private static Optional<Pair<CdsStructuredType, Map<String, Object>>>
      getOperationAsStructuredType(EventContext context) {
    List<CdsElementBuilder<?>> parameters =
        detectOperationParameters(context)
            .map(
                p ->
                    new CdsElementBuilder<>(
                            p.annotations().collect(Collectors.toList()),
                            p.getName(),
                            null,
                            false,
                            false,
                            p.isNotNull(),
                            false,
                            p.getDefaultValue().orElse(null),
                            "")
                        .type(p.getType()))
            .collect(Collectors.toList());

    if (!parameters.isEmpty()) {
      String typeName = context.getEvent().concat("_internalOperationType");
      CdsStructuredTypeBuilder<CdsStructuredType> builder =
          new CdsStructuredTypeBuilder<>(
              Collections.singletonList(
                  CdsAnnotationImpl.annotation(
                      CdsAnnotations.INTERNAL_OPERATION_TYPE.name(), true)),
              String.join(".", context.getService().getName(), typeName),
              typeName,
              CdsKind.TYPE,
              "");
      builder.addElements(parameters);
      CdsStructuredType type = builder.build();

      Map<String, Object> data = new HashMap<>();
      type.elements().forEach(e -> data.put(e.getName(), context.get(e.getName())));

      return Optional.of(Pair.of(builder.build(), data));
    }
    return Optional.empty();
  }

  private static Stream<CdsParameter> detectOperationParameters(EventContext context) {
    String targetName;
    String operationName;

    if (context.getTarget() != null) {
      targetName = context.getTarget().getQualifiedName();
      operationName = context.getEvent();
    } else {
      targetName = null;
      operationName = String.join(".", context.getService().getName(), context.getEvent());
    }

    return CdsModelUtils.getFunction(context.getModel(), targetName, operationName)
        .map(CdsFunction::parameters)
        .orElseGet(
            () ->
                CdsModelUtils.getAction(context.getModel(), targetName, operationName)
                    .map(CdsAction::parameters)
                    .orElseGet(Stream::empty));
  }
}
