/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messaging.service;

import static com.sap.cds.services.messaging.utils.CloudEventUtils.toJson;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiPredicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.environment.CdsProperties.Messaging.MessagingServiceConfig;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.messaging.MessagingErrorEventContext;
import com.sap.cds.services.messaging.MessagingService;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.messaging.utils.CloudEventUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;

/**
 * Implementation of the {@link MessagingService} interface.
 */
public abstract class AbstractMessagingService extends ServiceDelegator implements MessagingService {

	private static final Logger logger = LoggerFactory.getLogger(AbstractMessagingService.class);
	protected static final String FORMAT_CLOUDEVENTS = "cloudevents";
	public static final String CONTEXT_PARAMETERS_KEY = "cds.context.parameters";

	protected final MessagingServiceConfig serviceConfig;
	protected final CdsRuntime runtime;
	protected final MessageQueue queue;
	protected final boolean forceListening;
	private final boolean isStructured;

	protected AbstractMessagingService(MessagingServiceConfig serviceConfig, CdsRuntime runtime) {
		super(serviceConfig.getName());
		this.serviceConfig = serviceConfig;
		this.runtime = runtime;
		this.forceListening = serviceConfig.getQueue().isForceListening();
		this.isStructured = serviceConfig.isStructured();

		// create the initial queue representation by reading the configuration. Further the topics
		// will be completed by the handler registrations and application service configuration/model.
		queue = MessageQueue.create(serviceConfig, getTopicMatcher(), runtime.getEnvironment().getApplicationInfo());
	}

	/**
	 * Performs the initialization of the messaging service, by initializing the queues and registering the topic subscriptions.
	 */
	public void init() {
		if (runtime.getEnvironment().getCdsProperties().getEnvironment().getCommand().isEnabled()) {
			return;
		}
		createOrUpdateQueuesAndSubscriptions();
	}

	/**
	 * Used to stop the resources allocated by the messaging service, e.g. AMQP connections
	 */
	public void stop() {
		// empty by default
	}

	/**
	 * Performs the initialization of the messaging service, by initializing the queues and registering the topic subscriptions.
	 *
	 * @return true, if the initialization created queues and/or subscriptions
	 */
	protected boolean createOrUpdateQueuesAndSubscriptions() {
		logger.info("Initializing subscriptions of messaging service '{}'", getName());
		String queueName = toFullyQualifiedQueueName(queue);

		try {
			// check whether the queue should be reset on the broker
			if (runtime.getEnvironment().getCdsProperties().getMessaging().isResetQueues()) {
				try {
					createQueue(queueName, queue.getProperties());
					removeQueue(queueName);
					logger.warn("Reset the queue '{}' of service '{}'", queueName, getName());
				} catch (IOException e) { // NOSONAR
					logger.warn("Failed to reset queue '{}' of service '{}'", queueName, getName());
				}
			}

			if (!queue.getTopics().isEmpty() || forceListening) {
				// create the queue on the broker
				createQueue(queueName, queue.getProperties());
				logger.info("Created queue '{}' for service '{}'", queueName, getName());

				for(MessageTopic topic : queue.getTopics()) {
					String topicName = topic.getBrokerName();
					try {
						createQueueSubscription(queueName, topicName);
						logger.info("Subscribed topic '{}' on queue '{}' for service '{}'", topicName, queueName, getName());
					} catch (IOException e) {
						logger.error("Failed to subscribe topic '{}' on queue '{}' for service '{}'", topicName, queueName, getName(), e);
					}
				}
				// now register the queue listener
				try {
					registerQueueListener(queueName, new MessagingBrokerQueueListener(this, queueName, queue, runtime, this.isStructured));
				} catch (IllegalArgumentException | IOException e) {
					logger.error("Failed to register the listener to the queue '{}' for service '{}'", queueName, getName(), e);
				}
				return true;
			} else {
				logger.warn("There are no queue subscriptions available for the service '{}'", getName());
				return false;
			}
		} catch (Exception e) { // NOSONAR
			logger.error("Failed to create queue '{}' for service '{}'", queueName, getName(), e);
			return false;
		} finally {
			logger.debug("Finished initializing subscriptions of service '{}'", getName());
		}
	}

	protected boolean isCloudEventsFormat() {
		return serviceConfig.getFormat() != null && serviceConfig.getFormat().trim().equalsIgnoreCase(FORMAT_CLOUDEVENTS);
	}

	// MessagingService API and default event handler implementation

	@Override
	public void emit(String topic, String message) {
		emit(topic, message, null, null);
	}

	@Override
	public void emit(String topic, Map<String, Object> dataMap) {
		emit(topic, null, dataMap, null);
	}

	@Override
	public void emit(String topic, Map<String, Object> dataMap, Map<String, Object> headersMap) {
		emit(topic, null, dataMap, headersMap);
	}

	private void emit(String topic, String message, Map<String, Object> dataMap, Map<String, Object> headersMap) {
		TopicMessageEventContext context = TopicMessageEventContext.create(topic);
		retrieveContextParameters(headersMap).ifPresent(p -> p.forEach(context::put));

		if (isStructured) {
			if (message != null) {
				context.setDataMap(new HashMap<>(Map.of("message", message)));
				context.setHeadersMap(new HashMap<>());
			} else {
				context.setDataMap(dataMap);
				context.setHeadersMap(headersMap != null ? headersMap : new HashMap<>());
			}
		} else {
			if (message != null) {
				context.setData(message);
			} else {
				Map<String, Object> map;
				if (headersMap != null) {
					map = new HashMap<>(headersMap);
					map.put("data", new HashMap<>(dataMap));
				} else {
					map = dataMap;
				}

				context.setData(toJson(map));
			}
		}

		emit(context);
	}

	@SuppressWarnings("unchecked")
	private Optional<Map<String, Object>> retrieveContextParameters(Map<String, Object> headersMap) {
		if (headersMap != null) {
			return Optional.ofNullable((Map<String, Object>) headersMap.remove(CONTEXT_PARAMETERS_KEY));
		}
		return Optional.empty();
	}

	@On
	@HandlerOrder(OrderConstants.On.AUTO_COMPLETE)
	private void autoComplete(EventContext context) {
		// complete only for known topic registrations
		if (queue.hasEvent(context.getEvent()) || forceListening) {
			context.setCompleted();
		}
	}

	@Before
	@HandlerOrder(Integer.MIN_VALUE)
	protected void validateEventContext(TopicMessageEventContext context) {
		if (!Boolean.TRUE.equals(context.getIsInbound())) {
			if (context.getData() == null && context.getDataMap() == null && context.getHeadersMap() == null) {
				throw new ErrorStatusException(CdsErrorStatuses.NO_MESSAGE_PROVIDED);
			} else if (context.getDataMap() == null && context.getHeadersMap() != null) {
				context.setDataMap(new HashMap<>());
			} else if (context.getDataMap() != null && context.getHeadersMap() == null) {
				context.setHeadersMap(new HashMap<>());
			}
		}
	}

	@Before
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	protected void cloudEventsFormatter(TopicMessageEventContext context) {
		if (!Boolean.TRUE.equals(context.getIsInbound()) && isCloudEventsFormat()) {
			if(context.getHeadersMap() != null) {
				context.setHeadersMap(CloudEventUtils.toCloudEvent(context.getHeadersMap(), context.getEvent(), serviceConfig.getPublishPrefix()));
			} else {
				context.setData(CloudEventUtils.toCloudEvent(context.getData(), context.getEvent(), serviceConfig.getPublishPrefix()));
			}
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.PRIORITY)
	protected void sendMessageEvent(TopicMessageEventContext context) {
		// if not inbound message
		if (!Boolean.TRUE.equals(context.getIsInbound())) {
			AbstractMessagingService service = (AbstractMessagingService) context.getService();

			// get the event topic
			String topic = toFullyQualifiedTopicName(context.getEvent(), false);

			logger.debug("The service event '{}' is going to be emitted on service '{}' to topic '{}'", context.getEvent(), getName(), topic);

			service.emitTopicMessage(topic, context);
			context.setCompleted();
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void defaultErrorHandler(MessagingErrorEventContext context) {
		// by default we don't acknowledge the message
		context.setResult(false);
	}

	// Topic subscriptions based on event handlers

	@Override
	public void on(String[] events, String[] entities, int order, Handler handler) {
		super.on(events, entities, order, handler);

		Arrays.stream(events)
		.filter(event -> !StringUtils.isEmpty(event) && !event.equals("*") && !EVENT_MESSAGING_ERROR.equals(event))
		.forEach(event -> {
			// topics from event handlers are not assumed to be full-qualified (they are prefixed with the namespace)
			String topic = toFullyQualifiedTopicName(event, true);
			queue.addTopic(new MessageTopic(event, topic));
		});
	}

	// methods to be provided by specific messaging service implementation

	/**
	 * Translates the queue name corresponding the broker queue name specification.
	 *
	 * @param queue queue specification
	 * @return broker environment specific queue name
	 */
	protected String toFullyQualifiedQueueName(MessageQueue queue) {
		return queue.getName();
	}

	/**
	 * Translates the given event definition to the broker environment specific name.
	 *
	 * @param event event definition
	 * @param inbound determines whether topic is used for subscription
	 *
	 * @return the list of topic names corresponding the broker environment.
	 */
	protected String toFullyQualifiedTopicName(String event, boolean inbound) {
		if (inbound) {
			if (serviceConfig.getSubscribePrefix() != null) {
				return serviceConfig.getSubscribePrefix() + event;
			}
		} else {
			if (serviceConfig.getPublishPrefix() != null) {
				return serviceConfig.getPublishPrefix() + event;
			}
		}

		return event;
	}

	/**
	 * The topic matcher is used when the broker subscribes to a topic pattern.
	 * In this case the implementation is needed to match the incoming message topic to the CAP registered topic pattern (e.g "+/+/+/myTopic" to "my/client/namespace/myTopic").
	 *
	 * @return the topic matcher
	 */
	protected BiPredicate<MessageTopic, String> getTopicMatcher() {
		return (internalTopic, brokerTopic) -> Objects.equals(internalTopic.getBrokerName(), brokerTopic);
	}

	/**
	 * Request the broker for queue deletion with all its subscriptions.
	 *
	 * @param name queue name
	 * @throws IOException In case an error occurs while creating the queue
	 */
	protected abstract void removeQueue(String name) throws IOException;

	/**
	 * Request the broker for creating a queue with the specified queue name.
	 *
	 * @param name queue name
	 * @param properties queue configuration properties
	 * @throws IOException In case an error occurs while creating the queue
	 */
	protected abstract void createQueue(String name, Map<String, Object> properties) throws IOException;

	/**
	 * Requests the broker for creating a queue topic subscription. The method returns<code>true</code>
	 * only when the queue topic subscription was successfully created or already available.
	 * Otherwise <code>false</code> is returned.
	 *
	 * @param queue the queue name
	 * @param topic the topic the queue should subscribe to
	 * @throws IOException In case an error occurs during the subscription
	 */
	protected abstract void createQueueSubscription(String queue, String topic) throws IOException;

	/**
	 * Registers the {@link MessagingBrokerQueueListener} implementation to the specified queue of
	 * the message broker.
	 *
	 * @param queue the queue name
	 * @param listener implementation of the {@link MessagingBrokerQueueListener} interface
	 * @throws IOException In case when any errors occurs while registering the listener to the broker
	 *
	 */
	protected abstract void registerQueueListener(String queue, MessagingBrokerQueueListener listener) throws IOException;

	/**
	 * Performs the message emit on the messaging broker.
	 *
	 * @param topic the message topic
	 * @param topicMessageEventContext the TopicMessageEventContext of the message
	 */
	protected abstract void emitTopicMessage(String topic, TopicMessageEventContext topicMessageEventContext);

}
