/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messaging.service;

import static java.util.Collections.emptyMap;

import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.ServiceException;
import com.sap.cds.services.messaging.MessagingErrorEventContext;
import com.sap.cds.services.messaging.MessagingService;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.messaging.utils.CloudEventUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * This class provides the implementation of a message listener which is responsible for
 * passing the received message to the service layer in order to invoke the appropriate custom
 * handler
 */
public class MessagingBrokerQueueListener {

	private static final Logger logger = LoggerFactory.getLogger(MessagingBrokerQueueListener.class);

	private final MessagingService service;
	private final String queueName;
	private final MessageQueue queue;
	private final CdsRuntime runtime;
	private final boolean isStructured;

	public MessagingBrokerQueueListener(MessagingService service, String queueName, MessageQueue queue, CdsRuntime runtime, boolean isStructured) {
		this.service = service;
		this.queueName = queueName;
		this.queue = queue;
		this.runtime = runtime;
		this.isStructured = isStructured;
	}

	public String getQueueName() {
		return queueName;
	}

	public void receivedMessage(MessageAccess message) {
		logger.debug("Received message on service '{}' from topic '{}' and queue '{}'.", service.getName(), message.getBrokerTopic(), queueName);

		List<MessageTopic> topics = queue.findTopic(message.getBrokerTopic());
		AtomicBoolean acknowledge = new AtomicBoolean(true);

		try {
			for(MessageTopic topic : topics) {
				try {

					// run in privileged mode, as there is no user info in messaging
					runtime.requestContext().privilegedUser().run(req -> {
						try {
							TopicMessageEventContext context = getContext(message, topic.getEventName());
							logger.debug("The message 'id:{}' from topic '{}' on service '{}' is going to be emitted as a service event '{}'", context.getMessageId(), topic.getBrokerName(),  service.getName(), topic.getEventName());
							service.emit(context);
						} catch (Throwable th) { // NOSONAR
							MessagingErrorEventContext errorContext = MessagingErrorEventContext.create();
							ServiceException e = th instanceof ServiceException se ? se : new ServiceException(th);
							errorContext.setException(e);
							service.emit(errorContext);

							if (!errorContext.getResult()) {
								acknowledge.set(false);
							}

							throw th;
						}
					});
				} catch (Exception e) { // NOSONAR
					throw new ErrorStatusException(CdsErrorStatuses.EVENT_PROCESSING_FAILED, topic.getEventName(), service.getName(), queueName, e);
				}
			}
		} finally {
			if(acknowledge.get()) {
				message.acknowledge();
			}
		}
	}

	private TopicMessageEventContext getContext(MessageAccess message, String eventTopicName) {
		TopicMessageEventContext context = TopicMessageEventContext.create(eventTopicName);
		// write headers first, so that explicit context setter always win in case of conflicts
		for (Map.Entry<String, String> header : message.getTechnicalHeaders().entrySet()) {
			context.put(header.getKey(), header.getValue());
		}

		if (this.isStructured) {
			context.setDataMap(message.getDataMap());
			context.setHeadersMap(message.getHeadersMap());
		} else {
			context.setData(message.getMessage());
		}

		if (message.getId() == null) {
			String id = null;
			if (isStructured) {
				if (message.getHeadersMap().containsKey(CloudEventUtils.KEY_ID)) {
					id = (String) message.getHeadersMap().get(CloudEventUtils.KEY_ID);
				}
			}
			context.setMessageId(id);
		} else {
			context.setMessageId(message.getId());
		}


		context.setIsInbound(true);
		return context;
	}

	public static interface MessageAccess {

		/**
		 * @return the message ID, determined by the broker
		 */
		public String getId();

		/**
		 * @return the raw message String, as received from the broker
		 */
		public String getMessage();

		/**
		 * @return the topic, from which the message was received
		 */
		public String getBrokerTopic();

		/**
		 * Acknowledges the message at the broker
		 */
		public void acknowledge();

		/**
		 * @return the structured data part of the message, only used if the structured flag is set
		 */
		public Map<String, Object> getDataMap();

		/**
		 * @return the headers of the message, only used if the structured flag is set
		 */
		public Map<String, Object> getHeadersMap();

		/**
		 * @return the technical headers that are directly written to the {@link TopicMessageEventContext}
		 */
		public default Map<String, String> getTechnicalHeaders() {
			return emptyMap();
		}

	}
}
