/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messaging.utils;

import java.time.Instant;
import java.util.Map;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.impl.parser.token.Jsonizer;
import com.sap.cds.services.messaging.CloudEventMessageEventContext;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

/**
 * Utilities for CloudEvents according to SAP standards
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class CloudEventUtils {

	public static final String KEY_ID = "id";
	public static final String KEY_SPECVERSION = "specversion";
	public static final String KEY_DATACONTENTTYPE = "datacontenttype";
	public static final String KEY_TIME = "time";
	public static final String KEY_TYPE = "type";
	public static final String KEY_SOURCE = "source";
	public static final String KEY_DATA = "data";


	/**
	 * Enriches the given message with cloudevents headers if not yet present
	 *
	 * @param message the message to enrich
	 * @param defaultType the default type
	 * @param namespace the namespace to try to be interpreted as source
	 * @return enriched cloudevents message string
	 */
	public static String toCloudEvent(String message, String defaultType, String namespace) {
		Map<String, Object> messageMap = toMap(message);
		if(messageMap == null) {
			return message;
		}

		return toJson(toCloudEvent(messageMap, defaultType, namespace));
	}

	/**
	 * Enriches the given headers map with cloudevents headers if not yet present
	 *
	 * @param headers the headers to enrich
	 * @param defaultType the default type
	 * @param namespace the namespace to try to be interpreted as source
	 * @return enriched cloudevents headers
	 */
	public static Map<String, Object> toCloudEvent(Map<String, Object> headers, String defaultType, String namespace) {
		headers.computeIfAbsent(KEY_ID, (k) -> UUID.randomUUID().toString());
		headers.putIfAbsent(KEY_SPECVERSION, "1.0");
		headers.putIfAbsent(KEY_DATACONTENTTYPE, "application/json");
		headers.computeIfAbsent(KEY_TIME, (k) -> Instant.now().toString());
		headers.putIfAbsent(KEY_TYPE, defaultType);
		headers.computeIfAbsent(KEY_SOURCE, (k) -> getDefaultSource(namespace));
		return headers;
	}

	private static String getDefaultSource(String namespace) {
		String source = null;
		if(namespace != null) {
			String normalizedNamespace = StringUtils.trim(namespace.trim(), '/');
			if(normalizedNamespace.endsWith("/ce")) {
				normalizedNamespace = normalizedNamespace.substring(0, normalizedNamespace.length() - 3);
			}
			if(normalizedNamespace.endsWith("/-")) {
				normalizedNamespace = normalizedNamespace.substring(0, normalizedNamespace.length() - 2);
			}
			long slashCount = normalizedNamespace.chars().filter(ch -> ch == '/').count();
			if(slashCount == 2 || slashCount == 1) {
				source = "/" + normalizedNamespace;
			}
		}

		if(source == null) {
			source = "/default/sap.cap/" + getPID();
		}
		return source;
	}

	/**
	 * Creates the cloud event message event context from the messaging context.
	 *
	 * @param context messaging context
	 * @param eventName CDS event name
	 *
	 * @return the {@link CloudEventMessageEventContext}
	 */
	public static CloudEventMessageEventContext toCloudEventMessageContext(TopicMessageEventContext context, String eventName) {
		CloudEventMessageEventContext ctx = CloudEventMessageEventContext.create(eventName);

		if (context.getDataMap() != null) {
			// if data map is not null, we assume that the payload and headers are stored as structured data.
			copyEntries(ctx, context.getHeadersMap());
			ctx.put(KEY_DATA, context.getDataMap());
		} else {
			Map<String, Object> data = toMap(context.getData());

			if (data == null) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_CLOUDEVENTS_MESSAGE);
			} else {
				copyEntries(ctx, data);
			}
		}

		return ctx;
	}

	private static void copyEntries(CloudEventMessageEventContext ctx, Map<String, Object> map) {
		map.forEach((key, value) -> {
			if (key.equals(KEY_TIME)) {
				ctx.setTime(Instant.parse((String) value));
			} else {
				ctx.put(key, value);
			}
		});
	}

	/**
	 * Serializes the specified map to a JSON string.
	 *
	 * @param map map to be serialized in a JSON string
	 * @return the JSON string
	 */
	public static String toJson(Map<String, Object> map) {
		return Jsonizer.json(map);
	}

	/**
	 * Deserializes the specified JSON string to a map.
	 *
	 * @param jsonString JSON string to be deserialized to a map
	 * @return the map or null, if the JSON string is invalid
	 */
	public static Map<String, Object> toMap(String jsonString) {
		try {
			return JsonParser.map(JsonParser.parseJson(jsonString));
		} catch (Exception e) {
			return null;
		}
	}

	private static long getPID() {
		String processName = java.lang.management.ManagementFactory.getRuntimeMXBean().getName();
		if (!StringUtils.isEmpty(processName)) {
			try {
				return Long.parseLong(processName.split("@")[0]);
			}
			catch (Exception e) {
				return 0;
			}
		}
		return 0;
	}
}
