package com.sap.cds.services.messaging.utils;

import java.util.HashMap;
import java.util.Map;

import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.impl.parser.token.Jsonizer;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class MessagingOutboxUtils {

	/**
	 * Prefix for outbox message event targets.
	 * The messaging service name should be appended to this prefix.
	 */
	public static final String OUTBOX_MESSAGING_TARGET = "messaging/";

	private static final String DATA = "data";
	private static final String EVENT = "event";

	/**
	 * Creates the serialized outbox message for a messaging {@link EventContext}
	 *
	 * @param context the messaging {@link EventContext}
	 *
	 * @return the serialized outbox message
	 */
	public static String toOutboxMessage(EventContext context) {
		Map<String, Object> data = new HashMap<>();
		context.keySet().forEach(key -> data.put(key, context.get(key)));

		Map<String, Object> message = new HashMap<>();
		message.put(DATA, data);
		message.put(EVENT, context.getEvent());

		try {
			return Jsonizer.json(message);
		} catch (Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.CONTEXT_SERIALIZATION_FAILED, context.getEvent(), e);
		}
	}

	/**
	 * Creates the messaging {@link EventContext} from a serialized outbox message
	 *
	 * @param outboxContext the {@link OutboxMessageEventContext}
	 *
	 * @return the messaging {@link EventContext}
	 */
	@SuppressWarnings("unchecked")
	public static EventContext toEventContext(OutboxMessageEventContext outboxContext) {
		Map<String, Object> message;
		try {
			message = JsonParser.map(JsonParser.parseJson(outboxContext.getMessage()));
		} catch (Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.CONTEXT_DESERIALIZATION_FAILED, outboxContext.getEvent(), e);
		}

		EventContext context = EventContext.create((String) message.get(EVENT), null);
		Map<String, Object> data = (Map<String, Object>) message.get(DATA);
		data.forEach(context::put);

		return context;
	}

}
