/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messaging.service;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;

import com.sap.cds.services.environment.ApplicationInfo;
import com.sap.cds.services.environment.CdsProperties.Messaging.MessagingServiceConfig;
import com.sap.cds.services.messaging.utils.MessagingUtils;
import com.sap.cds.services.utils.StringUtils;


/**
 * This class represents the message queue topic subscription and provides also additional the
 * message type and the service the queue was created for.
 */
public class MessageQueue {

	private final String name;
	private final boolean fullyQualified;
	private final Map<String, Object> properties;
	private final Set<MessageTopic> topics = new HashSet<>();
	private final BiPredicate<MessageTopic, String> topicMatcher;

	static MessageQueue create(MessagingServiceConfig serviceConfig, BiPredicate<MessageTopic, String> topicMatcher, ApplicationInfo applicationInfo) {
		MessageQueue queue;
		Map<String, Object> properties = serviceConfig.getQueue().getConfig();
		if (StringUtils.isEmpty(serviceConfig.getQueue().getName())) {
			// generate the queue name
			String queueName = MessagingUtils.getQueueName(serviceConfig.getName(), applicationInfo);
			// the queue name is not yet full qualified, if brokers want they can append additional namespaces
			queue = new MessageQueue(queueName, false, properties, topicMatcher);
		} else {
			String queueName = serviceConfig.getQueue().getName();
			// take the queue name as is and assume it is full qualified (no additional namespaces appended)
			queue = new MessageQueue(queueName, true, properties, topicMatcher);
		}

		// get the configured topics from the service configuration
		// topics from the configuration are assumed to be full qualified (no additional namespaces appended)
		serviceConfig.getQueue().getSubscriptions().forEach(topic -> queue.addTopic(new MessageTopic(topic)));
		return queue;
	}

	private MessageQueue(String name, boolean fullyQualified, Map<String, Object> properties, BiPredicate<MessageTopic, String> topicMatcher) {
		this.name = name;
		this.fullyQualified = fullyQualified;
		this.properties = properties;
		this.topicMatcher = topicMatcher;
	}

	public String getName() {
		return name;
	}

	public boolean isFullyQualified() {
		return fullyQualified;
	}

	public Map<String, Object> getProperties() {
		return properties;
	}

	public List<MessageTopic> getTopics() {
		return topics.stream().collect(Collectors.toList());
	}

	public List<MessageTopic> findTopic(String topic) {
		List<MessageTopic> found = topics.stream().filter(t -> topicMatcher.test(t, topic)).collect(Collectors.toList());
		if(found.isEmpty()) {
			found.add(new MessageTopic(topic));
		}
		return found;
	}

	public boolean hasTopic(String topic) {
		return topics.stream().anyMatch(t -> topicMatcher.test(t, topic));
	}

	public boolean hasEvent(String event) {
		return topics.stream().anyMatch(t -> Objects.equals(t.getEventName(), event));
	}

	public void addTopic(MessageTopic topic) {
		topics.add(topic);
	}

}
