/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messaging.jms;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.environment.CdsProperties.Messaging.MessagingServiceConfig;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;


/**
 * This class manages the broker connections to the appropriate messaging services. By default there
 * is only one broker connection of each messaging kind. But if any configuration contains the custom
 * client configuration a new broker connection is created for the service.
 */
public abstract class BrokerConnectionProvider {

	private static final Logger logger = LoggerFactory.getLogger(BrokerConnectionProvider.class);

	private final String name;
	private Map<Map<String, String>, BrokerConnection> configuredConnections = new HashMap<>();

	protected BrokerConnectionProvider(String name) {
		this.name = name;
	}

	/**
	 * Initializes asynchronous the broker connection for the specified messaging service.  Notifies the consumer asynchronously
	 * when the broker connection is ready to use.
	 *
	 * @param config messaging service configuration
	 * @param consumer connection consumer
	 */
	public synchronized void asyncConnectionInitialization(MessagingServiceConfig config, Consumer<BrokerConnection> consumer) {
		try {
			final BrokerConnection connection = createConnection(config);
			// perform asynchronous initialization of the broker connection
			run(() -> {
				synchronized (connection) {
					// check whether to be connected or use shared
					if (!connection.isConnected()) {
						logger.info("Initializing messaging connection '{}'", connection.getName());
						try {
							connection.connect();
							logger.debug("Finished initializing messaging connection '{}'", connection.getName());

						} catch (Throwable th) {
							logger.warn("Could not initialize messaging connection '{}'", connection.getName(), th);
						}
					}
				}
				// the consumer initialization can be performed parallel (queue/topic subscriptions)
				consumer.accept(connection);
			});
		} catch (Throwable th) { // NOSONAR
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_BROKER_CONFIGURATION, config.getName(), th);
		}
	}

	/**
	 * Determines whether a new broker connection should be created for the specified service or the
	 * default one can be reused.
	 *
	 * @param config messaging service
	 * @return appropriate connection to the broker
	 */
	private BrokerConnection createConnection(MessagingServiceConfig config) {
		try {
			// separate the connection if configured as dedicated
			if (config.getConnection().isDedicated()) {
				return createBrokerConnection(config.getName(), config.getConnection().getProperties());
			}

			// otherwise check the connection properties in order to reuse the connection with the same properties
			BrokerConnection connection = configuredConnections.entrySet()
					.stream()
					.filter(e -> StringUtils.equalsMapIgnoreCase(e.getKey(), config.getConnection().getProperties()))
					.map(e -> e.getValue())
					.findFirst()
					.orElse(null);


			if (connection == null) {
				String connectionName = "shared-" + name + "-" + configuredConnections.size();
				connection = createBrokerConnection(connectionName, config.getConnection().getProperties());
				configuredConnections.put(config.getConnection().getProperties(), connection);
			}

			return connection;
		} catch (Throwable th) { // NOSONAR
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_BROKER_CONFIGURATION, name, th);
		}
	}

	private void run(Runnable action) {
		// asynchronous run
		Thread runner = new Thread(action, name + " - Initializer");
		runner.setDaemon(true);
		runner.start();
	}

	/**
	 * Creates the broker connection corresponding the specified client configuration properties.
	 *
	 * @param name connection name
	 * @param clientProperties client configuration properties
	 * @return broker connection
	 * @throws Exception when the connection cannot be created
	 */
	protected abstract BrokerConnection createBrokerConnection(String name, Map<String, String> clientProperties) throws Exception; // NOSONAR
}

