/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messaging.jms;

import static com.sap.cds.services.messaging.utils.MessagingUtils.toStringMessage;

import java.nio.charset.StandardCharsets;

import jakarta.jms.BytesMessage;
import jakarta.jms.Connection;
import jakarta.jms.JMSException;
import jakarta.jms.MessageProducer;
import jakarta.jms.Session;
import jakarta.jms.Topic;

import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * Represents the emitter session for outbound messages on a broker connection.
 *
 */
class MessageEmitter {

	private final Session session;
	private final MessageProducer producer;

	MessageEmitter(Connection connection) throws JMSException {
		this.session = connection.createSession();
		// create a null producer in order to send to any topic
		this.producer = session.createProducer(null);
	}

	public void emitTopicMessage(String topic, TopicMessageEventContext messageEventContext) {
		try {
			Topic jmsTopic = session.createTopic(topic);
			producer.send(jmsTopic, createByteMessage(toStringMessage(messageEventContext)));
		} catch (JMSException e) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_EMITTING_FAILED, topic, e);
		}
	}

	private BytesMessage createByteMessage(String content) throws JMSException {
		BytesMessage msg = session.createBytesMessage();
		msg.clearBody();
		msg.writeBytes(content.getBytes(StandardCharsets.UTF_8));
		return msg;
	}
}
