/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.messaging.jms;

import static com.sap.cds.services.messaging.utils.MessagingUtils.toStructuredMessage;

import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.impl.util.Pair;
import com.sap.cds.services.messaging.service.MessagingBrokerQueueListener;
import com.sap.cds.services.messaging.service.MessagingBrokerQueueListener.MessageAccess;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

import jakarta.jms.BytesMessage;
import jakarta.jms.Connection;
import jakarta.jms.JMSException;
import jakarta.jms.Message;
import jakarta.jms.MessageConsumer;
import jakarta.jms.Queue;
import jakarta.jms.Session;
import jakarta.jms.TextMessage;

/**
 * This class handles the queue connection session in order to receive the queue messages. The
 * receiving session is a separate thread which is blocked listening to the queue. In case of non-handled
 * messages the received JMS message is not acknowledged and the session is closed in order to
 * fail over the message to the next consumer.
 */
class MessageQueueReader {

	private static final Logger logger = LoggerFactory.getLogger(MessageQueueReader.class);

	private final String queueName;
	private final MessagingBrokerQueueListener listener;
	private final TopicAccessor topicAccessor;

	MessageQueueReader(String queueName, MessagingBrokerQueueListener listener, TopicAccessor topicAccessor) {
		this.topicAccessor = topicAccessor;
		this.queueName = queueName;
		this.listener = listener;
	}

	public void startListening(Connection connection) throws JMSException {
		new MessageQueueReaderThread(connection).start();
	}

	private class MessageQueueReaderThread extends Thread {

		private final Connection connection;
		private Session session;
		private MessageConsumer consumer;

		public MessageQueueReaderThread(Connection connection) throws JMSException {
			super(queueName + " - Listener");
			this.connection = connection;
			initSession();
		}

		private void initSession() throws JMSException {
			// TODO: check if initSession() is the correct approach to release the message
			// Auto-Acknowledgment with throwing Message Listener didn't work
			// session.recover() didn't work
			// Configuring the Redelivery Policy didn't work
			// All tested approaches only redispatched the message internally, without involving the broker
			// Transacted sessions aren't supported by MQ and EM
			// What is the correct approach for this in JMS?
			Session prevSession = this.session;
			try {
				this.session = connection.createSession(Session.CLIENT_ACKNOWLEDGE);
				Queue queue = session.createQueue(queueName);
				this.consumer = session.createConsumer(queue);
			} finally {
				if (prevSession != null) {
					prevSession.close();
				}
			}
		}

		@Override
		public void run() {
			try {
				while(true) {
					Message jmsMessage = consumer.receive();
					// terminate the thread, if the consumer was closed
					if (jmsMessage == null) {
						logger.debug("The message consumer of the queue reader '{}' was closed", queueName);
						break;
					}

					ReceivedMessage message = null;
					try {
						message = new ReceivedMessage(jmsMessage, topicAccessor);
						listener.receivedMessage(message);
					} catch (Throwable th) {
						if (message != null) {
							logger.error("The received message of the queue '{}' and topic '{}' could not be handled", queueName,  !StringUtils.isEmpty(message.getBrokerTopic()) ? message.getBrokerTopic() : "???", th);
						} else {
							logger.error("The received message of the queue '{}' could not be handled", queueName, th);
						}
						// recreate the session in order to release the message for other consumer.
						initSession();
					}
				}
			} catch (JMSException e) {
				logger.error("The queue reader '{}' was interrupted", queueName, e);
			}
		}

	}

	/**
	 *
	 * Extract the message and topic string from the JMS message.
	 *
	 */
	private static class ReceivedMessage implements MessageAccess {

		private final Message jmsMessage;
		private final String message;
		private final String topic;
		private final String id;
		private Map<String, Object> dataMap;
		private Map<String, Object> headersMap;

		public ReceivedMessage(Message message, TopicAccessor topicAccessor) throws JMSException {
			this.id = message.getJMSMessageID();
			this.jmsMessage = message;

			if (message instanceof TextMessage txtMsg) {
				// get the message text
				this.message = txtMsg.getText();
				this.topic = topicAccessor.getFromTopic(message);
			} else if (message instanceof BytesMessage byteMsg) {
				byte[] byteData = new byte[(int) byteMsg.getBodyLength()];
				byteMsg.readBytes(byteData);
				byteMsg.reset();
				// get the message text
				this.message = new String(byteData, StandardCharsets.UTF_8);
				this.topic = topicAccessor.getFromTopic(message);
			} else {
				throw new JMSException("Unknown event message format: " + message.getClass().getName());
			}
		}

		@Override
		public String getMessage() {
			return message;
		}

		@Override
		public Map<String, Object> getDataMap() {
			if (this.dataMap == null) {
				populateMaps();
			}

			return this.dataMap;
		}

		@Override
		public Map<String, Object> getHeadersMap() {
			if (this.headersMap == null) {
				populateMaps();
			}

			return this.headersMap;
		}

		private void populateMaps() {
			Pair<Map<String, Object>, Map<String, Object>> maps = toStructuredMessage(this.message);

			this.dataMap = maps.left;
			this.headersMap = maps.right;
		}

		@Override
		public String getId() {
			return id;
		}

		@Override
		public String getTenant() {
			// the default implementation for JMS does not expect the tenant. If in future any JMS based messaging
			// implementation provides the tenant we can introduce a TennantAccessor like for topics the TopicAccessor.
			return null;
		}

		@Override
		public String getBrokerTopic() {
			return topic;
		}

		@Override
		public void acknowledge() {
			try {
				jmsMessage.acknowledge();
			} catch (JMSException e) {
				throw new ErrorStatusException(CdsErrorStatuses.ACKNOWLEDGMENT_FAILED, topic, e);
			}
		}
	}
}
