/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.messaging.jms;

import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.messaging.service.MessagingBrokerQueueListener;
import jakarta.jms.Connection;
import jakarta.jms.ConnectionFactory;
import jakarta.jms.JMSException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Represents the broker connection for outbound (emitter) and inbound (queues) messages. The
 * connection can be shared by several services of the same technical service instance.
 */
public class BrokerConnection {

  private static final Logger logger = LoggerFactory.getLogger(BrokerConnection.class);
  private static final Timer REESTABLISH_TIMER = new Timer("Broker Connection Reestablisher", true);
  private static final int MIN_REESTABLISH_DELAY = 120000; // 2 min
  private static final int MAX_REESTABLISH_DELAY = 600000; // 10 min

  private final String name;
  private final ConnectionFactory connectionFactory;
  private final List<MessageQueueReader> queueReaders = new ArrayList<>();
  private volatile Connection connection;
  private volatile MessageEmitter emitter;
  private volatile boolean isConnected;
  private volatile int reestablishDelay = MIN_REESTABLISH_DELAY;

  /**
   * @param name name of the connection
   * @param connectionFactory JMS connection factory
   */
  public BrokerConnection(String name, ConnectionFactory connectionFactory) {
    this.name = name;
    this.connectionFactory = connectionFactory;
  }

  /**
   * Connects to the destination broker and creates the emitter session on the broker connection.
   *
   * @throws IOException when any errors occurs
   */
  public void connect() throws IOException {
    logger.debug("Opening broker connection '{}'", name);
    if (!isConnected) {
      AtomicBoolean preventScheduling = new AtomicBoolean();
      try {
        connection = connectionFactory.createConnection();
        connection.setExceptionListener(
            e -> {
              logger.warn("The broker connection '{}' failed", name);
              if (!preventScheduling.getAndSet(true)) {
                scheduleReconnect();
              }
            });

        // get the message emitter session for outbound messages
        emitter = new MessageEmitter(connection);

        connection.start();
        isConnected = true;
        reestablishDelay = MIN_REESTABLISH_DELAY;

        logger.info("The messaging broker connection '{}' has been established", name);
      } catch (JMSException e) {
        if (!preventScheduling.getAndSet(true)) {
          scheduleReconnect();
        }
        throw new IOException("Could not establish broker connection '" + name + "'", e);
      }
    } else {
      logger.warn("The broker connection '{}' is already connected", name);
    }
  }

  /**
   * Closes the connection to the broker.
   *
   * @throws JMSException if the JMS provider fails to close the connection due to some internal
   *     error.
   */
  public void close() throws JMSException {
    try {
      if (connection != null) {
        logger.debug("Closing broker connection '{}'", name);
        // this will close all underlying sessions, consumers and producers
        connection.close();
      }
    } finally {
      isConnected = false;
    }
  }

  private void scheduleReconnect() {
    logger.debug(
        "The broker connection '{}' reconnect attempt will be scheduled in {} ms",
        name,
        reestablishDelay);
    REESTABLISH_TIMER.schedule(
        new TimerTask() {
          @Override
          public void run() {
            try {
              close();
            } catch (JMSException e) {
              logger.debug("Failed to close broker connection '{}' before reconnecting", name, e);
            }

            try {
              connect();
              // reestablish reader threads
              for (MessageQueueReader queueReader : queueReaders) {
                queueReader.startListening(connection);
              }
            } catch (JMSException | IOException e) {
              logger.error("Failed to reestablish broker connection '{}'", name, e);
            }
          }
        },
        reestablishDelay);
    reestablishDelay = Math.min(reestablishDelay * 2, MAX_REESTABLISH_DELAY);
  }

  /**
   * Registers the queue listener for the specified queue.
   *
   * @param queue the queue name
   * @param listener the queue listeners
   * @param topicAccessor the topic accessor in order to extract the destination topic from the
   *     delivered message.
   * @throws IOException when any errors occurs
   */
  public void registerQueueListener(
      String queue, MessagingBrokerQueueListener listener, TopicAccessor topicAccessor)
      throws IOException {
    try {
      logger.debug("Registering queue listener on '{}'", queue);
      MessageQueueReader queueReader = new MessageQueueReader(queue, listener, topicAccessor);
      queueReaders.add(queueReader);
      queueReader.startListening(connection);
    } catch (JMSException e) {
      throw new IOException(e.getMessage(), e);
    }
  }

  /**
   * Emits the given message to the specified topic
   *
   * @param topic topic
   * @param messageEventContext the message context
   */
  public void emitTopicMessage(String topic, TopicMessageEventContext messageEventContext) {
    logger.debug("Emitting a message to topic '{}'", topic);
    emitter.emitTopicMessage(topic, messageEventContext);
  }

  /**
   * Returns the connection name.
   *
   * @return connection name
   */
  public String getName() {
    return name;
  }

  /**
   * Determines whether the connection is opened.
   *
   * @return <code>true</code> if opened and <code>false</code> otherwise.
   */
  public boolean isConnected() {
    return isConnected;
  }
}
