/*
 * © 2020-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.messaging.service;

import static java.util.Collections.emptyMap;

import com.sap.cds.services.ServiceException;
import com.sap.cds.services.environment.CdsProperties.Messaging.MessagingServiceConfig;
import com.sap.cds.services.messaging.MessagingErrorEventContext;
import com.sap.cds.services.messaging.MessagingService;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.messaging.utils.CloudEventUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.CorrelationIdUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.lib.mt.TenantUtils;
import com.sap.cds.services.utils.outbox.OutboxUtils;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class provides the implementation of a message listener which is responsible for passing the
 * received message to the service layer in order to invoke the appropriate custom handler
 */
public class MessagingBrokerQueueListener {

  private static final Logger logger = LoggerFactory.getLogger(MessagingBrokerQueueListener.class);

  private final MessagingService service;
  private final String queueName;
  private final MessageQueue queue;
  private final CdsRuntime runtime;

  @Deprecated(forRemoval = true, since = "4.0.0")
  public MessagingBrokerQueueListener(
      MessagingService service,
      String queueName,
      MessageQueue queue,
      CdsRuntime runtime,
      boolean structured) {
    this(service, new MessagingServiceConfig(), queueName, queue, runtime);
  }

  public MessagingBrokerQueueListener(
      MessagingService service,
      MessagingServiceConfig serviceConfig,
      String queueName,
      MessageQueue queue,
      CdsRuntime runtime) {

    if (serviceConfig.getInbox().isEnabled()) {
      this.service = OutboxUtils.outboxed(service, serviceConfig.getInbox().getName(), runtime);
    } else {
      this.service = service;
    }

    this.queueName = queueName;
    this.queue = queue;
    this.runtime = runtime;
  }

  public String getQueueName() {
    return queueName;
  }

  public void receivedMessage(MessageAccess message) {
    logger.debug(
        "Received message on service '{}' from topic '{}' and queue '{}'.",
        service.getName(),
        message.getBrokerTopic(),
        queueName);

    List<MessageTopic> topics = queue.findTopic(message.getBrokerTopic());
    AtomicBoolean acknowledge = new AtomicBoolean(true);
    AtomicBoolean error = new AtomicBoolean(false);

    try {
      for (MessageTopic topic : topics) {
        try {
          // run in privileged mode, as there is no user info in messaging
          runtime
              .requestContext()
              .systemUser(message.getTenant())
              .privilegedUser()
              .modifyParameters(
                  params -> {
                    // correlation ID from the message headers if available
                    if (message.getHeadersMap() != null
                        && message
                            .getHeadersMap()
                            .containsKey(CorrelationIdUtils.CORRELATION_ID_FIELD)) {
                      params.setCorrelationId(
                          (String)
                              message.getHeadersMap().get(CorrelationIdUtils.CORRELATION_ID_FIELD));
                    }
                  })
              .run(
                  req -> {
                    try {
                      TopicMessageEventContext context = getContext(message, topic.getEventName());
                      logger.debug(
                          "The message 'id:{}' from topic '{}' on service '{}' is going to be emitted as a service event '{}'",
                          context.getMessageId(),
                          topic.getBrokerName(),
                          service.getName(),
                          topic.getEventName());
                      service.emit(context);
                    } catch (Throwable th) { // NOSONAR
                      // on errors the error handling is performed in the tenant context
                      error.set(true);
                      performErrorHandling(th, acknowledge, message);

                      throw th;
                    }
                  });
        } catch (Throwable th) { // NOSONAR
          if (!error.get()) {
            // if the tenant request context cannot be created (unknown tenant) we perform the error
            // handling in the
            // provider tenant context
            logger.debug(
                "The tenant request context for '{}' cannot be created", message.getTenant());

            if (TenantUtils.isUnknownTenant(th)) {
              th =
                  new ErrorStatusException(
                      CdsErrorStatuses.TENANT_NOT_EXISTS, message.getTenant(), th);
            }

            performErrorHandling(th, acknowledge, message);
          }

          throw new ErrorStatusException(
              CdsErrorStatuses.EVENT_PROCESSING_FAILED,
              topic.getEventName(),
              service.getName(),
              queueName,
              th);
        }
      }
    } finally {
      if (acknowledge.get()) {
        message.acknowledge();
      }
    }
  }

  private void performErrorHandling(
      Throwable th, AtomicBoolean acknowledge, MessageAccess message) {
    MessagingErrorEventContext errorContext = MessagingErrorEventContext.create();
    ServiceException e = th instanceof ServiceException se ? se : new ServiceException(th);
    errorContext.setException(e);
    errorContext.setTenant(message.getTenant());
    errorContext.setMessageHeaders(message.getHeadersMap());
    errorContext.setMessageData(message.getDataMap());

    service.emit(errorContext);

    if (!errorContext.getResult()) {
      acknowledge.set(false);
    }
  }

  private TopicMessageEventContext getContext(MessageAccess message, String eventTopicName) {
    TopicMessageEventContext context = TopicMessageEventContext.create(eventTopicName);
    // write headers first, so that explicit context setter always win in case of conflicts
    for (Map.Entry<String, String> header : message.getTechnicalHeaders().entrySet()) {
      context.put(header.getKey(), header.getValue());
    }

    context.setDataMap(message.getDataMap());
    context.setHeadersMap(message.getHeadersMap());

    if (message.getId() == null) {
      String id = null;
      if (message.getHeadersMap().containsKey(CloudEventUtils.KEY_ID)) {
        id = (String) message.getHeadersMap().get(CloudEventUtils.KEY_ID);
      }
      context.setMessageId(id);
    } else {
      context.setMessageId(message.getId());
    }

    context.setIsInbound(true);
    return context;
  }

  public static interface MessageAccess {

    /**
     * @return the message ID, determined by the broker
     */
    public String getId();

    /**
     * @return the tenant ID provided from the message
     */
    public String getTenant();

    /**
     * @return the raw message String, as received from the broker
     */
    public String getMessage();

    /**
     * @return the topic, from which the message was received
     */
    public String getBrokerTopic();

    /** Acknowledges the message at the broker */
    public void acknowledge();

    /**
     * @return the structured data part of the message, only used if the structured flag is set
     */
    public Map<String, Object> getDataMap();

    /**
     * @return the headers of the message, only used if the structured flag is set
     */
    public Map<String, Object> getHeadersMap();

    /**
     * @return the technical headers that are directly written to the {@link
     *     TopicMessageEventContext}
     */
    public default Map<String, String> getTechnicalHeaders() {
      return emptyMap();
    }
  }
}
