/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.services.messaging.utils;

import static com.sap.cds.services.messaging.utils.CloudEventUtils.toJson;
import static com.sap.cds.services.messaging.utils.CloudEventUtils.toMap;

import com.sap.cds.impl.util.Pair;
import com.sap.cds.services.environment.ApplicationInfo;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessagingUtils {

  private static final Logger logger = LoggerFactory.getLogger(MessagingUtils.class);

  /**
   * Generates an application/service specific queue name.
   *
   * @param service full qualified service name
   * @param config application properties
   * @return unique app/service specific queue name
   */
  public static String getQueueName(String service, ApplicationInfo config) {
    StringBuilder queue = new StringBuilder(normalizeName(config.getName()));

    int lastIdx = service.lastIndexOf('.');
    String name;
    String qualifier;
    if (lastIdx > 0) {
      name = service.substring(lastIdx + 1);
      qualifier = service.substring(0, lastIdx);
    } else {
      name = service;
      qualifier = "";
    }

    queue.append('/');
    queue.append(getShortHash(config.getId()));
    queue.append('/');
    queue.append(normalizeName(name));
    queue.append('/');
    queue.append(getShortHash(qualifier));

    return queue.toString();
  }

  public static String toStringMessage(TopicMessageEventContext context) {
    Map<String, Object> messageMap = new HashMap<>(context.getHeadersMap());
    messageMap.put("data", context.getDataMap());
    return toJson(messageMap);
  }

  @SuppressWarnings("unchecked")
  public static Pair<Map<String, Object>, Map<String, Object>> toStructuredMessage(String message) {
    Map<String, Object> dataMap;
    Map<String, Object> headersMap;

    if (message == null) {
      dataMap = new HashMap<>();
      headersMap = new HashMap<>();
    } else {
      Map<String, Object> map = toMap(message);

      if (map != null) {
        Object data = map.get("data");
        if (data instanceof Map) {
          dataMap = (Map<String, Object>) map.remove("data");
          headersMap = map;
        } else if (data == null && map.containsKey("data")) {
          dataMap = new HashMap<>();
          headersMap = map;
          headersMap.remove("data");
        } else {
          dataMap = map;
          headersMap = new HashMap<>();
        }
      } else {
        dataMap = new HashMap<>(Map.of("message", message));
        headersMap = new HashMap<>();
      }
    }

    return Pair.of(dataMap, headersMap);
  }

  private static String normalizeName(String name) {
    return name.replaceAll("[^a-zA-Z0-9\\*\\.]", "").replace('.', '/');
  }

  /**
   * Calculates the MD5 hash from the specified string and truncates to 8 characters.
   *
   * @param text
   * @return 4 characters MD5 hash
   */
  private static String getShortHash(String text) {
    try {
      MessageDigest md = MessageDigest.getInstance("MD5");
      md.update(text.getBytes(StandardCharsets.UTF_8));
      byte[] digest = md.digest();
      StringBuffer sb = new StringBuffer();
      for (int i = 0; i < digest.length && i < 2; i++) {
        sb.append("%02x".formatted(digest[i] & 0xff));
      }
      return sb.toString();
    } catch (NoSuchAlgorithmException e) {
      logger.error("Could not get MD5 algorithm", e);
    }
    return text;
  }
}
