/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Utility class for the draft handling.
 *
 */
public class DraftUtils {

	private static final Logger log = LoggerFactory.getLogger(DraftUtils.class);

	/**
	 * Checks if the service has a draft enabled entity.
	 * @param serviceName the service name
	 * @param model the model
	 * @return {@code true} if the model has a draft enabled entity
	 */
	public static boolean isDraftEnabled(String serviceName, CdsModel model) {
		return model.entities()
				.anyMatch(e -> e.getQualifiedName().startsWith(serviceName) && isDraftEnabled(e));
	}

	/**
	 * Checks if the entity is draft enabled.
	 * @param entity the entity
	 * @return {@code true} if the entity is draft enabled
	 */
	public static boolean isDraftEnabled(CdsEntity entity) {
		return CdsAnnotations.DRAFT_ANNOTATION.isTrue(entity) ||
				CdsAnnotations.DRAFT_PREPARE_ANNOTATION.getOrDefault(entity) != null;
	}

	public static boolean isDraftEnabledNoChild(CdsEntity entity) {
		return CdsAnnotations.DRAFT_ANNOTATION.isTrue(entity);
	}

	public static void gcDraftsOfAllServices(CdsRuntime runtime) {
		Stream<DraftService> draftServices = runtime.getServiceCatalog().getServices(DraftService.class);
		draftServices.forEach(draftService -> {
			draftService.gcDrafts();
		});
	}

	public static void gcDraftsOfAllServicesAndTenants(CdsRuntime runtime) {
		TenantProviderService tenantProvider = runtime.getServiceCatalog().getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);
		List<String> tenants = tenantProvider.readTenants();
		if (tenants.isEmpty()) {
			tenants = Arrays.asList((String) null);
		}
		tenants.forEach(tenant -> {
			try {
				runtime.requestContext().privilegedUser().modifyUser(user -> user.setTenant(tenant)).recalculateFeatureToggles().run(context -> {
					gcDraftsOfAllServices(runtime);
				});
			} catch (Throwable t) {
				log.error("Failed to gc drafts of tenant '{}'", tenant, t);
			}
		});
	}

}
