/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;

public class TenantAwareCache<T, V> {

	private final Supplier<String> tenantSupplier;
	private final Supplier<T> valueSupplier;
	private final Supplier<V> validatorSupplier;
	private final Map<String, CacheEntry<T, V>> cached = new ConcurrentHashMap<>();

	public static <T> TenantAwareCache<T, CdsModel> create(Supplier<T> valueSupplier, CdsRuntime runtime) {
		Supplier<String> tenantSupplier = () -> RequestContext.isActive() ? RequestContext.getCurrent(runtime).getUserInfo().getTenant() : runtime.getProvidedUserInfo().getTenant();
		return new TenantAwareCache<>(tenantSupplier, valueSupplier, () -> RequestContext.getCurrent(runtime).getModel());
	}

	public static <T, V> TenantAwareCache<T, V> create(Supplier<String> tenantSupplier, Supplier<T> valueSupplier, Supplier<V> validatorSupplier) {
		return new TenantAwareCache<>(tenantSupplier, valueSupplier, validatorSupplier);
	}

	private TenantAwareCache(Supplier<String> tenantSupplier, Supplier<T> valueSupplier, Supplier<V> validatorSupplier) {
		this.tenantSupplier = tenantSupplier;
		this.valueSupplier = valueSupplier;
		this.validatorSupplier = validatorSupplier;
	}

	public T findOrCreate() {
		String tenantName = tenantSupplier.get();
		String tenantKey = tenantName == null ? "" : tenantName.trim();

		V validator = validatorSupplier.get();
		CacheEntry<T, V> entry = cached.get(tenantKey);
		if (entry != null) {
			if (!Objects.equals(validator, entry.validator)) {
				cached.remove(tenantKey);
				entry = null;
			}
		}
		if (entry == null) {
			T newValue = valueSupplier.get();
			CacheEntry<T, V> newEntry = new CacheEntry<T, V>(newValue, validator);
			if ((entry = cached.putIfAbsent(tenantKey, newEntry)) == null) {
				entry = newEntry;
			}
		}

		return entry.value;
	}

	private static class CacheEntry<T, V> {
		private final T value;
		private final V validator;

		public CacheEntry(T value, V validator) {
			this.value = value;
			this.validator = validator;
		}
	}
}
