/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.services.environment.ServiceBinding;
import com.sap.cds.services.runtime.CdsRuntime;

public class XsuaaUtils {

	private static final String XSUAA = "xsuaa";

	private final CdsRuntime runtime;

	public XsuaaUtils(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	/**
	 * @return	All found XSUAA bindings. In case a specific service binding is chosen via cds.security.xsuaa.binding,
	 * 			the list contains only this binding if available.
	 */
	public List<ServiceBinding> getXsuaaServiceBindings() {
		Stream<ServiceBinding> xsuaaBindings = runtime.getEnvironment().getServiceBindings().filter(b -> b.matches(XSUAA, XSUAA));

		String xsuaaServiceBinding = runtime.getEnvironment().getCdsProperties().getSecurity().getXsuaa().getBinding();
		if (!StringUtils.isEmpty(xsuaaServiceBinding)) {
			xsuaaBindings = xsuaaBindings.filter(b -> xsuaaServiceBinding.equalsIgnoreCase(b.getName()));
		}

		return xsuaaBindings.collect(Collectors.toList());
	}

	/**
	 * @return	{@code true} if condition for XSUAA authentication is given by application configuration, {@code false} otherwise.
	 */
	public boolean xsuaaConfiguredCondition() {
		// Property 'cds.security.xsuaa.enabled' controls XSUAA UserInfo integration
		return runtime.getEnvironment().getCdsProperties().getSecurity().getXsuaa().isEnabled() && !getXsuaaServiceBindings().isEmpty();
	}

}
