/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.messaging;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.services.environment.ApplicationInfo;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class MessagingUtils {

	private static final Logger logger = LoggerFactory.getLogger(MessagingUtils.class);

	/**
	 * Generates an application/service specific queue name.
	 *
	 * @param service full qualified service name
	 * @param config application properties
	 * @return unique app/service specific queue name
	 */
	public static String getQueueName(String service, ApplicationInfo config) {
		StringBuilder queue = new StringBuilder(normalizeName(config.getName()));

		int lastIdx = service.lastIndexOf('.');
		String name;
		String qualifier;
		if(lastIdx > 0) {
			name = service.substring(lastIdx + 1);
			qualifier = service.substring(0, lastIdx);
		} else {
			name = service;
			qualifier = "";
		}

		queue.append('/');
		queue.append(getShortHash(config.getId()));
		queue.append('/');
		queue.append(normalizeName(name));
		queue.append('/');
		queue.append(getShortHash(qualifier));

		return queue.toString();
	}

	/**
	 * Generates an application/service specific queue name.
	 *
	 * @param service CDS service definition
	 * @param config application properties
	 * @return unique app/service specific queue name
	 */
	public static String getQueueName(CdsService service, ApplicationInfo config) {
		StringBuilder queue = new StringBuilder(normalizeName(config.getName()));

		queue.append('/');
		queue.append(getShortHash(config.getId()));
		queue.append('/');
		queue.append(normalizeName(service.getName()));
		queue.append('/');
		queue.append(getShortHash(service.getQualifier()));

		return queue.toString();
	}

	/**
	 * Generates an service/entity/event specific topic name.
	 *
	 * @param service CDS service definition
	 * @param entity CDS entity definition
	 * @param event event
	 *
	 * @return the service/entity/event specific topic name
	 */
	public static String getTopicPattern(CdsService service, CdsEntity entity, String event) {
		StringBuilder topic = new StringBuilder("CAP/");

		topic.append(normalizeName(service.getName()));
		topic.append('/');
		topic.append(getShortHash(service.getQualifier()));

		if (entity != null) {
			topic.append('/');
			topic.append(normalizeName(entity.getName()));
		}

		topic.append('/');
		topic.append(normalizeName(event));

		return topic.toString();

	}

	/**
	 * Generates an service/entity/event specific topic name.
	 *
	 * @param service the full qualified service name
	 * @param entity CDS entity definition
	 * @param event event
	 *
	 * @return the service/entity/event specific topic name
	 */
	public static String getTopicPattern(String service, CdsEntity entity, String event) {
		return getTopicPattern(service, entity != null ? entity.getName() : null, event);
	}

	/**
	 * Generates an service/entity/event specific topic name.
	 *
	 * @param service the full qualified service name
	 * @param entity CDS entity definition
	 * @param event event
	 *
	 * @return the service/entity/event specific topic name
	 */
	public static String getTopicPattern(String service, String entity, String event) {
		StringBuilder topic = new StringBuilder("CAP/");

		int lastIdx = service.lastIndexOf('.');
		String name;
		String qualifier;
		if(lastIdx >= 0) {
			name = service.substring(lastIdx + 1);
			qualifier = service.substring(0, lastIdx);
		} else {
			name = service;
			qualifier = "";
		}

		topic.append(normalizeName(name));
		topic.append('/');
		topic.append(getShortHash(qualifier));

		if (entity != null) {

			// check full qualified
			lastIdx = entity.lastIndexOf('.');
			if (lastIdx >= 0) {
				name = entity.substring(lastIdx + 1);
				qualifier = entity.substring(0, lastIdx);

				if (!qualifier.equals(service)) {
					throw new ErrorStatusException(CdsErrorStatuses.MISDIRECTED_ENTITY, service, entity);
				}

				topic.append('/');
				topic.append(normalizeName(name));
			} else {
				topic.append('/');
				topic.append(normalizeName(entity));
			}
		}

		topic.append('/');
		topic.append(normalizeName(event));

		return topic.toString();

	}

	public static String normalizeName(String name) {
		return name.replaceAll("[^a-zA-Z0-9\\*\\.]", "").replace('.', '/');
	}

	/**
	 * Calculates the MD5 hash from the specified string and truncates to 8 characters.
	 *
	 * @param text
	 * @return 4 characters MD5 hash
	 */
	private static String getShortHash(String text) {
		try {
			MessageDigest md = MessageDigest.getInstance("MD5");
			md.update(text.getBytes(StandardCharsets.UTF_8));
			byte[] digest = md.digest();
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < digest.length && i < 2; i++) {
				sb.append(String.format("%02x", digest[i] & 0xff));
			}
			return sb.toString();
		} catch (NoSuchAlgorithmException e) {
			logger.error("Could not get MD5 algorithm", e);
		}
		return text;
	}
}
