/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.messaging.jms;

import java.io.IOException;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.JMSException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.utils.messaging.service.MessagingBrokerQueueListener;

/**
 * Represents the broker connection for outbount (emitter) and inbound (queues) messages. The
 * connection can be shared by several services of the same  technical service instance.
 *
 */
public class BrokerConnection {

	private static final Logger logger = LoggerFactory.getLogger(BrokerConnection.class);

	private final String name;
	private final ConnectionFactory connectionFactory;
	private volatile Connection connection;
	private volatile MessageEmitter emitter;
	private volatile boolean isConnected;

	/**
	 * @param name name of the connection
	 * @param connectionFactory JMS connection factory
	 */
	public BrokerConnection(String name, ConnectionFactory connectionFactory) {
		this.name = name;
		this.connectionFactory = connectionFactory;
	}

	/**
	 * Connects to the destination broker and creates the emitter session on the broker connection.
	 *
	 * @throws IOException when any errors occurs
	 */
	public void connect() throws IOException {
		logger.debug("Opening the broker connection '{}'", name);
		if (!isConnected) {
			try {
				connection = connectionFactory.createConnection();
				connection.setExceptionListener(e -> {
					logger.error("The broker connection '{}' is in trouble!", name, e);
					isConnected = false;
				});

				// get the message emitter session for outbound messages
				emitter = new MessageEmitter(connection);

				connection.start();
				isConnected = true;

				logger.info("The messaging broker connection '{}' has been established.", name);
			} catch (JMSException e) {
				throw new IOException("Broker connection lost", e);
			}
		} else {
			logger.error("The broker connection '{}' is already connected!", name);
		}
	}

	/**
	 * Closes the connection to the broker.
	 *
	 * @throws JMSException if the JMS provider fails to close the connection due to some internal error.
	 */
	public void close() throws JMSException {
		logger.debug("Closing the broker connection '{}'", name);
		connection.close();
		isConnected = false;
	}

	/**
	 * Registers the queue listener for the specified queue.
	 *
	 * @param queue the queue name
	 * @param listener the queue listeners
	 * @param topicAccessor the topic accessor in order to extract the destination topic from the
	 * 						delivered message.
	 * @param maxFailedCount the amount of attempts to treat the message before the message is going
	 * 						 to be acknowledged. If the value is null the amount of attempts is unlimited.
	 *
	 * @throws IOException when any errors occurs
	 */
	public void registerQueueListener(String queue, MessagingBrokerQueueListener listener, TopicAccessor topicAccessor, int maxFailedCount) throws IOException {
		try {
			logger.debug("Registering the queue listener on '{}'", queue);
			new MessageQueueReader(queue, listener, connection, topicAccessor, maxFailedCount).startListening();
		} catch (JMSException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	/**
	 * Emits the given message to the specified topic
	 *
	 * @param topic topic
	 * @param message raw message
	 */
	public void emitTopicMessage(String topic, String message) {
		logger.debug("Emitting a message to the topic '{}'", topic);
		emitter.emitTopicMessage(topic, message);
	}

	/**
	 * Returns the connection name.
	 *
	 * @return connection name
	 */
	public String getName() {
		return name;
	}

	/**
	 * Determines whether the connection is opened.
	 *
	 * @return <code>true</code> if opened and <code>false</code> otherwise.
	 */
	public boolean isConnected() {
		return isConnected;
	}
}
