/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.messaging.jms;

import java.nio.charset.StandardCharsets;

import javax.jms.BytesMessage;
import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.MessageProducer;
import javax.jms.Session;
import javax.jms.Topic;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 * Represents the emitter session for outbound messages on a broker connection.
 *
 */
class MessageEmitter {

	private static final Logger logger = LoggerFactory.getLogger(MessageEmitter.class);

	private final Session session;
	private final MessageProducer producer;

	MessageEmitter(Connection connection) throws JMSException {
		this.session = connection.createSession();
		// create a null producer in order to send to any topic
		this.producer = session.createProducer(null);
	}

	public void emitTopicMessage(String topic, String message) {
		try {
			Topic jmsTopic = session.createTopic(topic);
			producer.send(jmsTopic, createByteMessage(message));
			logger.debug("Emitting message to topic '{}'", topic);
		} catch (JMSException e) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_EMITTING_FAILED, topic, e);
		}
	}

	private BytesMessage createByteMessage(String content) throws JMSException {
		BytesMessage msg = session.createBytesMessage();
		msg.clearBody();
		msg.writeBytes(content.getBytes(StandardCharsets.UTF_8));
		return msg;
	}
}
