/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.messaging.jms;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import javax.jms.BytesMessage;
import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.Queue;
import javax.jms.Session;
import javax.jms.TextMessage;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.messaging.service.MessagingBrokerQueueListener;

/**
 * This class handles the queue connection session in order to receive the queue messages. The
 * receiving session is a separate thread which is blocked listening to the queue. In case of non-handled
 * messages the received JMS message is not acknowledged and the session is closed in order to
 * fail over the message to the next consumer.
 */
class MessageQueueReader implements Runnable {

	private static final Logger logger = LoggerFactory.getLogger(MessageQueueReader.class);

	private TopicAccessor topicAccessor;
	private MessagingBrokerQueueListener listener;
	private String queueName;
	private Connection connection;
	private Session session;
	private Queue queue;
	private MessageConsumer consumer;
	private int maxFaildAttempts;

	private Map<String, Integer> failedCounter = new HashMap<>();

	MessageQueueReader(String queueName, MessagingBrokerQueueListener listener, Connection connection, TopicAccessor topicAccessor, int maxFaildAttempts) throws JMSException {
		this.maxFaildAttempts = maxFaildAttempts;
		this.topicAccessor = topicAccessor;
		this.queueName = queueName;
		this.listener = listener;
		this.connection = connection;
		this.session = connection.createSession(false, Session.CLIENT_ACKNOWLEDGE);
		this.queue = session.createQueue(queueName);
		this.consumer = session.createConsumer(this.queue);
	}

	public void startListening() {
		// start the listening thread
		(new Thread(this, queueName + " - Listener")).start();
	}

	@Override
	public void run() {
		try {
			while(consumer != null) {
				Message message = consumer.receive();
				if (message != null) {
					ReceivedMessage textMessage = null;
					try {
						textMessage = new ReceivedMessage(message, topicAccessor);
						listener.receivedMessage(textMessage.getMessage(), textMessage.getTopic(), message.getJMSMessageID());
						message.acknowledge();
					} catch (JMSException e) {
						logger.warn("Failed to parse JMS message on queue '{}'", queueName, e);
						checkUnhandledMessage(message);
					} catch (Throwable th) {
						if (textMessage != null) {
							logger.error("The received message of the queue '{}' and topic '{}' could not be handled!", queueName,  !StringUtils.isEmpty(textMessage.getTopic()) ? textMessage.getTopic() : "???", th);
						} else {
							logger.error("The received message of the queue '{}' could not be handled!", queueName, th);
						}
						checkUnhandledMessage(message);
					}
				}
			}
		} catch (JMSException e) {
			logger.error("The queue reader '{}' was interrupted!", queueName, e);
		}
	}

	private void checkUnhandledMessage(Message message) throws JMSException {

		if (maxFaildAttempts > 0) {
			if (!failedCounter.containsKey(message.getJMSMessageID())) {
				failedCounter.put(message.getJMSMessageID(), 0);
			}

			int failedAttempts = failedCounter.get(message.getJMSMessageID()) + 1;

			// check whether the may try is reached
			if (failedAttempts >= maxFaildAttempts) {
				logger.debug("Auto-acknowledged message with ID '{}', as maximum failed attempts were reached", message.getJMSMessageID());
				message.acknowledge();
				failedCounter.remove(message.getJMSMessageID());
			} else {
				failedCounter.put(message.getJMSMessageID(), failedAttempts);
				// recreate the session in order to release the message for other consumer.
				Session prevSession = session;
				try {
					session = connection.createSession(false, Session.CLIENT_ACKNOWLEDGE);
					queue = session.createQueue(queueName);
					consumer = session.createConsumer(queue);
				} finally {
					prevSession.close();
				}
			}
		}
	}

	/**
	 *
	 * Extract the message and topic string from the JMS message.
	 *
	 */
	private static class ReceivedMessage {

		private String message;
		private String topic;

		public ReceivedMessage(Message message, TopicAccessor topicAccessor) throws JMSException {
			super();

			if (message instanceof TextMessage) {
				TextMessage txtMsg = (TextMessage) message;
				// get the message text
				this.message = txtMsg.getText();
				this.topic = topicAccessor.getFromTopic(message);
			} else if (message instanceof BytesMessage) {
				BytesMessage byteMsg = (BytesMessage) message;
				byte[] byteData = new byte[(int) byteMsg.getBodyLength()];
				byteMsg.readBytes(byteData);
				byteMsg.reset();

				// get the message text
				this.message = new String(byteData, StandardCharsets.UTF_8);
				this.topic = topicAccessor.getFromTopic(message);
			} else {
				throw new JMSException("Unknown event message format: " + message.getClass().getName());
			}
		}

		public String getMessage() {
			return message;
		}

		public String getTopic() {
			return topic;
		}
	}
}
