/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.messaging.service;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.reflect.CdsEvent;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.environment.CdsProperties.Messaging.MessagingServiceConfig;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.messaging.MessagingService;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;

/**
 * Implementation of the {@link MessagingService} interface.
 */
public abstract class AbstractMessagingService extends ServiceDelegator implements MessagingService {

	private static final Logger logger = LoggerFactory.getLogger(AbstractMessagingService.class);
	private static final ObjectMapper mapper = new ObjectMapper();
	private static final String IS_OUTBOXED = "IS_OUTBOXED";

	protected final MessagingServiceConfig serviceConfig;
	protected final CdsRuntime runtime;
	protected final MessageQueue queue;

	private final boolean outboxed;


	protected AbstractMessagingService(MessagingServiceConfig serviceConfig, CdsRuntime runtime) {
		super(serviceConfig.getName());

		this.serviceConfig = serviceConfig;
		this.runtime = runtime;

		// create the initial queue representation by reading the configuration. Further the topics
		// will be completed by the handler registrations and application service configuration/model.
		queue = MessageQueue.create(serviceConfig, runtime.getEnvironment().getApplicationInfo());
		outboxed = serviceConfig.getOutbox().isEnabled();
	}

	/**
	 * Performs the initialization of the messaging service, by initializing the queues and registering the topic subscriptions.
	 */
	public void init() {
		createOrUpdateQueuesAndSubscriptions();
	}

	/**
	 * Performs the initialization of the messaging service, by initializing the queues and registering the topic subscriptions.
	 *
	 * @return true, if the initialization created queues and/or subscriptions
	 */
	protected boolean createOrUpdateQueuesAndSubscriptions() {
		logger.info("Initializing subscriptions of messaging service '{}'", getName());
		String queueName = toFullyQualifiedQueueName(queue);

		try {
			// check whether the queue should be reset on the broker
			if (runtime.getEnvironment().getCdsProperties().getMessaging().isResetQueues()) {
				try {
					createQueue(queueName, queue.getProperties());
					removeQueue(queueName);
					logger.warn("Reset the queue '{}' of service '{}'", queueName, getName());
				} catch (IOException e) { // NOSONAR
					logger.warn("Failed to reset queue '{}' of service '{}'", queueName, getName());
				}
			}

			if (!queue.getTopics().isEmpty()) {
				// create the queue on the broker
				createQueue(queueName, queue.getProperties());
				logger.info("Created queue '{}' for service '{}'", queueName, getName());

				for(MessageTopic topic : queue.getTopics()) {
					String topicName = topic.getBrokerName();
					try {
						createQueueSubscription(queueName, topicName);
						logger.info("Subscribed topic '{}' on queue '{}' for service '{}'", topicName, queueName, getName());
					} catch (IOException e) {
						logger.error("Failed to subscribe topic '{}' on queue '{}' for service '{}'", topicName, queueName, getName(), e);
					}
				}
				// now register the queue listener
				try {
					registerQueueListener(queueName, new MessagingBrokerQueueListener(this, queueName, queue));
				} catch (IllegalArgumentException | IOException e) {
					logger.error("Failed to register the listener to the queue '{}' for service '{}'", queueName, getName(), e);
				}

				return true;
			} else {
				logger.warn("There are no queue subscriptions available for the service '{}'", getName());
				return false;
			}
		} catch (Exception e) { // NOSONAR
			logger.error("Failed to create queue '{}' for service '{}'", queueName, getName(), e);
			return false;
		} finally {
			logger.debug("Finished initializing subscriptions of service '{}'", getName());
		}
	}

	@Override
	public void emit(EventContext context) {
		// check whether the outbox should be used
		if (shouldBeOutboxed(context)) {
			OutboxService outbox = runtime.getServiceCatalog().getService(OutboxService.class, OutboxService.DEFAULT_NAME);
			// mark as outboxed context
			context.put(IS_OUTBOXED, true);
			outbox.enroll(this, context);
		} else {
			super.emit(context);
		}
	}

	private boolean isInbound(EventContext context) {
		return context instanceof TopicMessageEventContext && ((TopicMessageEventContext) context).getIsInbound();
	}

	private boolean shouldBeOutboxed(EventContext context) {
		// outbox is on and the context is not from the outbox and also not inbound message
		return outboxed && context.get(IS_OUTBOXED) == null && !isInbound(context);
	}

	// MessagingService API and default event handler implementation

	@Override
	public void emit(String topic, String message) {
		TopicMessageEventContext context = TopicMessageEventContext.create(topic);
		context.setData(message);
		emit(context);
	}

	@Override
	public void emit(String topic, Map<String, Object> message) {
		emit(topic, toJson(message));
	}

	@On
	@HandlerOrder(OrderConstants.On.AUTO_COMPLETE)
	private void autoComplete(EventContext context) {
		// complete only for known topic registrations
		if (queue.hasEvent(context.getEvent())) {
			context.setCompleted();
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.MESSAGING)
	protected void sendMessageEvent(TopicMessageEventContext context) {
		// if not inbound message
		if (!Boolean.TRUE.equals(context.getIsInbound())) {
			AbstractMessagingService service = (AbstractMessagingService) context.getService();

			// check whether the specified event is a declared model event or technical topic
			String topic = context.getModel()
					.findEvent(context.getEvent())
					.map(this::toTopicName)
					.orElse(context.getEvent());

			service.emitTopicMessage(toFullyQualifiedTopicName(topic), context.getData());
			context.setCompleted();
		}
	}

	private String toJson(Object object) {
		try {
			return mapper.writeValueAsString(object);
		} catch (JsonProcessingException e) {
			throw new ErrorStatusException(ErrorStatuses.SERVER_ERROR, e);
		}
	}

	// Topic subscriptions based on event handlers

	@Override
	public void on(String[] events, String[] entities, int order, Handler handler) {
		super.on(events, entities, order, handler);
		checkEvents(events);
	}

	private void checkEvents(String[] events) {
		Arrays.stream(events)
		.filter(event -> !StringUtils.isEmpty(event) && !event.equals("*"))
		// topics from event handlers are not assumed to be full-qualified (they are prefixed with the namespace)
		.forEach(event -> {
			// check whether the specified event is a declared model event or technical topic
			String topic = runtime.getCdsModel().findEvent(event)
					.map(this::toTopicName)
					.orElse(event);

			queue.addTopic(new MessageTopic(event, toFullyQualifiedTopicName(topic)));
		});
	}

	// methods to be provided by specific messaging service implementation

	/**
	 * Translates the specified CDS event into the broker topic structure.
	 *
	 * @param event CDS declared event
	 * @return broker agnostic topic specification
	 */
	protected String toTopicName(CdsEvent event) {
		return event.getQualifiedName().replace('.', '/');
	}

	/**
	 * Translates the queue name corresponding the broker queue name specification.
	 *
	 * @param queue queue specification
	 * @return broker environment specific queue name
	 */
	protected String toFullyQualifiedQueueName(MessageQueue queue) {
		return queue.getName();
	}

	/**
	 * Translates the given topic definition to the broker environment specific name (in the enterprise
	 * messaging this could be extended by the client name space)
	 *
	 * @param topic broker specific topic definition
	 *
	 * @return the topic name corresponding the broker environment.
	 */
	protected String toFullyQualifiedTopicName(String topic) {
		return topic;
	}

	/**
	 * Request the broker for queue deletion with all its subscriptions.
	 *
	 * @param name queue name
	 * @throws IOException In case an error occurs while creating the queue
	 */
	protected abstract void removeQueue(String name) throws IOException;

	/**
	 * Request the broker for creating a queue with the specified queue name.
	 *
	 * @param name queue name
	 * @param properties queue configuration properties
	 * @throws IOException In case an error occurs while creating the queue
	 */
	protected abstract void createQueue(String name, Map<String, String> properties) throws IOException;

	/**
	 * Requests the broker for creating a queue topic subscription. The method returns<code>true</code>
	 * only when the queue topic subscription was successfully created or already available.
	 * Otherwise <code>false</code> is returned.
	 *
	 * @param queue the queue name
	 * @param topic the topic the queue should subscribe to
	 * @throws IOException In case an error occurs during the subscription
	 */
	protected abstract void createQueueSubscription(String queue, String topic) throws IOException;

	/**
	 * Registers the {@link MessagingBrokerQueueListener} implementation to the specified queue of
	 * the message broker.
	 *
	 * @param queue the queue name
	 * @param listener implementation of the {@link MessagingBrokerQueueListener} interface
	 * @throws IOException In case when any errors occurs while registering the listener to the broker
	 *
	 */
	protected abstract void registerQueueListener(String queue, MessagingBrokerQueueListener listener) throws IOException;

	/**
	 * Performs the message emit on the messaging broker.
	 *
	 * @param topic the message topic
	 * @param message the raw string message
	 */
	protected abstract void emitTopicMessage(String topic, String message);

}
