/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.messaging.service;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.sap.cds.services.environment.ApplicationInfo;
import com.sap.cds.services.environment.CdsProperties.Messaging.MessagingServiceConfig;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.messaging.MessagingUtils;


/**
 * This class represents the message queue topic subscription and provides also additional the
 * message type and the service the queue was created for.
 */
public class MessageQueue {

	private final String name;
	private final boolean fullyQualified;
	private final Map<String, String> properties;
	private final Set<MessageTopic> topics = new HashSet<>();

	static MessageQueue create(MessagingServiceConfig serviceConfig, ApplicationInfo applicationInfo) {
		MessageQueue queue;
		Map<String, String> properties = serviceConfig.getQueue().getConfig();
		if (StringUtils.isEmpty(serviceConfig.getQueue().getName())) {
			// generate the queue name
			String queueName = MessagingUtils.getQueueName(serviceConfig.getName(), applicationInfo);
			// the queue name is not yet full qualified, if brokers want they can append additional namespaces
			queue = new MessageQueue(queueName, false, properties);
		} else {
			String queueName = serviceConfig.getQueue().getName();
			// take the queue name as is and assume it is full qualified (no additional namespaces appended)
			queue = new MessageQueue(queueName, true, properties);
		}

		// get the configured topics from the service configuration
		// topics from the configuration are assumed to be full qualified (no additional namespaces appended)
		serviceConfig.getQueue().getSubscriptions().forEach(topic -> queue.addTopic(new MessageTopic(topic)));
		return queue;
	}

	private MessageQueue(String name, boolean fullyQualified, Map<String, String> properties) {
		this.name = name;
		this.fullyQualified = fullyQualified;
		this.properties = properties;
	}

	public String getName() {
		return name;
	}

	public boolean isFullyQualified() {
		return fullyQualified;
	}

	public Map<String, String> getProperties() {
		return properties;
	}

	public List<MessageTopic> getTopics() {
		return topics.stream().collect(Collectors.toList());
	}

	public MessageTopic findTopic(String topic) {
		return topics.stream().filter(t -> Objects.equals(t.getBrokerName(), topic)).findFirst().orElse(new MessageTopic(topic));
	}

	public boolean hasEvent(String topic) {
		return topics.stream().filter(t -> Objects.equals(t.getEventName(), topic)).count() > 0;
	}

	public void addTopic(MessageTopic topic) {
		topics.add(topic);
	}

}
