/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.model;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsAnnotation;

/**
 * Supported CDS annotations
 */
public enum CdsAnnotations {

	RESTRICT(null, "restrict"),
	REQUIRES(null, "requires"),

	READONLY(false,	"readonly"),
	INSERTONLY(false, "insertonly"),

	INSERTABLE(true, "Capabilities.Insertable",	"Capabilities.InsertRestrictions.Insertable"),
	UPDATABLE(true,	"Capabilities.Updatable", "Capabilities.UpdateRestrictions.Updatable"),
	DELETABLE(true,	"Capabilities.Deletable", "Capabilities.DeleteRestrictions.Deletable"),

	AUTOEXPOSED(false, "cds.autoexposed"), // indicates that an entity is auto exposed (explicitly or automatically)
	AUTOEXPOSE(false, "cds.autoexpose"), // used to explicitly auto expose an entity

	ON_INSERT(null, "cds.on.insert", "odata.on.insert"),
	ON_UPDATE(null, "cds.on.update", "odata.on.update"),

	MANDATORY(false, "mandatory", "Common.FieldControl.Mandatory", "FieldControl.Mandatory"),
	FIELD_CONTROL_READONLY(false, "Common.FieldControl.ReadOnly", "FieldControl.ReadOnly"),
	COMMON_FIELDCONTROL(null, "Common.FieldControl"),
	ASSERT_RANGE(false, "assert.range"),
	ASSERT_FORMAT(false, "assert.format"),
	ASSERT_NOTNULL(true, "assert.notNull"),

	QUERY_LIMIT_DEFAULT(null, "cds.query.limit.default", "cds.query.limit"),
	QUERY_LIMIT_MAX(null, "cds.query.limit.max"),
	DEFAULT_ORDER(null, "cds.default.order", "odata.default.order"),

	CORE_COMPUTED(false, "Core.Computed"),
	CORE_IMMUTABLE(false, "Core.Immutable"),
	CORE_IS_MEDIA_TYPE(null, "Core.IsMediaType"),
	CORE_MEDIA_TYPE(null, "Core.MediaType"),
	CORE_CONTENT_DISPOSITION_FILENAME(null, "Core.ContentDisposition.Filename"),

	SINGLETON(false, "odata.singleton"),

	DRAFT_ANNOTATION(false, "odata.draft.enabled"),
	DRAFT_PREPARE_ANNOTATION(null, "Common.DraftNode.PreparationAction"),

	ODATA_FOREIGN_KEY_FOR(null, "odata.foreignKey4"),

	PATH(null, "endpoints.path", "path"),
	PROTOCOLS(null, "endpoints.protocol", "protocols"),
	ENDPOINTS(null, "endpoints"),
	IGNORE(false, "cds.ignore"),
	SERVE_IGNORE(false, "cds.serve.ignore"),

	MESSAGING(false, "cds.messaging"),
	MESSAGING_TYPE(null, "cds.messaging.type"),
	MESSAGING_EMIT(null, "cds.messaging.emit"),
	MESSAGING_SKIP(false, "cds.messaging.skip"),

	PERSISTENCE_NAME(null, "cds.persistence.name"),
	ETAG(false,"odata.etag"),

	AGGREGATION_DEFAULT(null, "Aggregation.default"),
	SEMANTICS_CURRENCY_CODE(false, "Semantics.currencyCode"),
	SEMANTICS_UNIT_OF_MEASURE(false, "Semantics.unitOfMeasure");

	private final String[] names;
	private final Object defaultValue;

	private CdsAnnotations(Object defaultValue, String... names) {
		this.defaultValue = defaultValue;
		this.names = names; // NOSONAR
	}

	@SuppressWarnings("unchecked")
	public <T> T getOrDefault(CdsAnnotatable cdsModelElement) {
		return (T) getOrValue(cdsModelElement, this.defaultValue);
	}

	public boolean isTrue(CdsAnnotatable cdsModelElement) {
		return Boolean.TRUE.equals(getOrDefault(cdsModelElement));
	}

	@SuppressWarnings("unchecked")
	public <T> List<T> getListOrDefault(CdsAnnotatable cdsModelElement) {
		return (List<T>) getListOrValue(cdsModelElement, this.defaultValue);
	}

	@SuppressWarnings("unchecked")
	public <T> List<T> getListOrValue(CdsAnnotatable cdsModelElement, T value) {

		Object annotationValue = getOrValue(cdsModelElement, value);
		if (annotationValue != null) {
			if (annotationValue instanceof List) {
				return (List<T>)annotationValue;
			} else {
				return (List<T>) Arrays.asList(annotationValue);
			}
		}
		return null;
	}

	public <T> T getOrValue(CdsAnnotatable cdsModelElement, T value) {

		for (String name : names) {
			Optional<CdsAnnotation<T>> annotation = cdsModelElement.findAnnotation(name);
			if (annotation.isPresent()) {
				T annotationValue = annotation.get().getValue();
				if (annotationValue != null && !annotationValue.equals(value)) {
					return annotationValue;

				} else if(annotationValue == null && value != null) {
					return annotationValue;
				}
			}
		}

		return value;
	}
}
