/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.model;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.reflect.CdsAction;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsDefinitionNotFoundException;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsElementNotFoundException;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsFunction;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

public class CdsModelUtils {

	private final static ObjectMapper mapper = new ObjectMapper()
			.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
			.configure(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY, true);

	private CdsModelUtils() {
		// no instances
	}

	/**
	 * Returns a {@link Restriction} instance reflecting @restrict resp. @requires on the given service or {@code null}
	 * if not annotated.
	 *
	 * @param cdsModelElement	The CDS model element to be inspected
	 * @return	The {@link Restriction} instance
	 */
	public static Restriction getRestrictionOrNull(CdsAnnotatable cdsModelElement) {

		Restriction result = null;

		List<Object> privs = CdsAnnotations.RESTRICT.getOrDefault(cdsModelElement);
		if (privs != null) {
			result = new Restriction();
			Privilege[] privileges = mapper.convertValue(privs, Privilege[].class);
			result.addPrivileges(privileges);
		}

		Object annotationRoles = CdsAnnotations.REQUIRES.getOrDefault(cdsModelElement);
		if (annotationRoles != null) {
			if (result == null) {
				result = new Restriction();
			}
			String[] roles = mapper.convertValue(annotationRoles, String[].class);
			for (String role : roles) {
				result.addPrivilege( new Privilege().addGrant(Privilege.PredefinedGrant.ALL).addRole(role) );
			}
		}

		return result; // could be null if there are no privileges
	}

	/**
	 * Returns {@code true} if the passed model element (either service or entity) has no effective restriction.
	 * Please note that this does not include hierarchical dependencies.
	 *
	 * @param cdsModelElement	The element from the CDS model
	 * @return	{@code true} if public
	 */
	public static boolean isPublic(CdsAnnotatable cdsModelElement) {

		// model element is public if there is either no restriction or a restriction containing a 'wildcard all' privilege
		Restriction restriction = CdsModelUtils.getRestrictionOrNull(cdsModelElement);
		if (restriction != null) {
			return restriction.privileges().filter(priv -> {
				return priv.getGrants().contains(Privilege.PredefinedGrant.ALL.toString()) &&
						priv.getRoles().contains(Privilege.PredefinedRole.ANY_USER.toString()) &&
						(StringUtils.isEmpty(priv.getWhereCondition()) || !priv.getWhereCondition().contains("$user"));
			}).findAny().isPresent();
		}
		return true;
	}

	/**
	 * Returns a list of all events that are not restricted for the given model element, although the element itself is restricted in general.
	 *
	 * @param cdsModelElement	The CDS model element
	 * @return	The list of public events
	 */
	public static List<String> getPublicEvents(CdsAnnotatable cdsModelElement) {

		Set<String> publicEvents = new HashSet<>();
		Restriction restriction = CdsModelUtils.getRestrictionOrNull(cdsModelElement);
		if (restriction != null) {
			publicEvents = restriction.privileges().filter(priv -> {
				return priv.getRoles().contains(Privilege.PredefinedRole.ANY_USER.toString()) &&
						(StringUtils.isEmpty(priv.getWhereCondition()) || !priv.getWhereCondition().contains("$user"));
			}).flatMap(priv -> priv.getGrants().stream()).collect(Collectors.toSet());
		}
		if (publicEvents.contains(Privilege.PredefinedGrant.ALL.toString())) {
			publicEvents = Collections.singleton(Privilege.PredefinedGrant.ALL.toString()); // '*' dominates all others
		}
		return publicEvents.stream().collect(Collectors.toList());
	}

	public static AnalysisResult getEntityPath(CqnStructuredTypeRef ref, CdsModel model) {
		try {
			return CqnAnalyzer.create(model).analyze(ref);
		} catch (CdsElementNotFoundException e) {
			throw new ErrorStatusException(CdsErrorStatuses.ELEMENT_NOT_FOUND, e.getElementName(),
					e.getDefinition().getName(), e);
		} catch (CdsDefinitionNotFoundException e) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_FOUND, e.getQualifiedName(), e);
		}
	}

	public static boolean isChildEntity(CqnStructuredTypeRef ref, CdsModel model) {
		CdsEntity entity = null;
		for(Segment s: ref.segments()) {
			if (entity == null) {
				entity = model.getEntity(s.id());
			} else {
				String name = entity.getQualifiedName();
				CdsElement association = entity.findAssociation(s.id())
						.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.ASSOCIATION_NOT_FOUND, s.id(), name));
				if (!association.getType().as(CdsAssociationType.class).isComposition()) {
					return false;
				}
			}
		}
		return true;
	}

	public static CdsEntity getTargetEntity(CqnSelect select, CdsModel model) {
		return getEntityPath(select, model).targetEntity();
	}

	public static CdsEntity getTargetEntity(CqnStructuredTypeRef ref, CdsModel model) {
		return getEntityPath(ref, model).targetEntity();
	}

	public static AnalysisResult getEntityPath(CqnSelect select, CdsModel model) {
		return getEntityPath(select.ref(), model);
	}

	public static CdsEntity getRefTarget(CqnStructuredTypeRef ref, CdsEntity target) {
		CdsEntity refTarget = target;
		for(Segment s : ref.segments()) {
			if(s.id().equals("*")) {
				return null; // can't handle *
			}
			refTarget = refTarget.getTargetOf(s.id());
		}
		return refTarget;
	}

	public static String getTargetKeysAsString(CdsModel model, CqnStatement statement) {
		AnalysisResult analysis = CqnAnalyzer.create(model).analyze(statement.ref());
		String keyValues = analysis.targetKeyValues().entrySet().stream() //
				.map(entry -> entry.getKey() + "=" + entry.getValue()) //
				.collect(Collectors.joining(","));
		return keyValues;
	}

	public static Optional<CdsFunction> findFunction(CdsService service, CdsEntity entity, String name) {
		Optional<CdsFunction> result = service.functions().filter(f -> f.getName().equals(name)).findAny();
		if (entity != null && !result.isPresent()) {
			return entity.functions().filter(f -> f.getName().equals(name)).findAny();
		}
		return result;
	}

	public static Optional<CdsAction> findAction(CdsService service, CdsEntity entity, String name) {
		Optional<CdsAction> result = service.actions().filter(a -> a.getName().equals(name)).findAny();
		if (entity != null && !result.isPresent()) {
			return entity.actions().filter(a -> a.getName().equals(name)).findAny();
		}
		return result;
	}

}
