/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.model;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.cqn.CqnModifier;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnStatement;

public class CqnUtils {

	private CqnUtils() {
		// hidden
	}

	/**
	 * Converts the object array to an indexed map.
	 * The first object has the key <code>0</code>, the second <code>1</code>, etc.
	 *
	 * @param paramValues the objects
	 * @return the indexed map
	 */
	public static Map<String, Object> convertToIndexMap(Object... paramValues) {
		Map<String, Object> parameters = new HashMap<>();
		for (int i = 0; i < paramValues.length; i++) {
			parameters.put(String.valueOf(i), paramValues[i]);
		}
		return parameters;
	}

	/**
	 * Creates a {@link CqnStatement} with a modified where clause according to the passed function.
	 *
	 * @param <S> The type of the statement
	 *
	 * @param cqn	The original {@link CqnStatement}.
	 * @param modifyWhere	A function that maps an existing {@link Predicate} to the predicate in the modified statement.
	 * @return	The modified {@link CqnStatement}.
	 */
	public static <S extends CqnStatement> S modifiedWhere(final S cqn, Function<Predicate, Predicate> modifyWhere) {
		return CQL.copy(cqn, new WherePredicateModifier(modifyWhere));
	}

	// Defined as concrete class to prevent extensive creation of anonymous classes
	private static class WherePredicateModifier implements CqnModifier {

		private final Function<Predicate, Predicate> modifyWhere;

		WherePredicateModifier(Function<Predicate, Predicate> modifyWhere) {
			this.modifyWhere = modifyWhere;
		}

		@Override
		public Predicate where(Predicate pred) {
			return modifyWhere.apply(pred);
		}
	}

	/**
	 * Creates a function which combines an additional {@link Predicate} to the original predicate with a logical AND.
	 *
	 * @param pred	The {@link Predicate} to combine with AND.
	 * @return	The function
	 */
	public static Function<Predicate, Predicate> andPredicate(CqnPredicate pred) {
		return new AndPredicate(pred);
	}

	// Defined as concrete class to prevent extensive creation of anonymous classes
	private static class AndPredicate implements Function<Predicate, Predicate> {

		private CqnPredicate rhsPred; // could be null

		AndPredicate(CqnPredicate rhsPred) {
			this.rhsPred = rhsPred;
		}

		@Override
		public Predicate apply(Predicate lhsPred) {
			return lhsPred != null && rhsPred != null ?	CQL.and(lhsPred, rhsPred) :
				lhsPred != null ? lhsPred : CQL.copy(rhsPred); // NOSONAR
		}

	}
}


