/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.sap.cds.services.utils.StringUtils;

/**
 * A {@code Privilege} specifies access rules for operations.
 * A Where clause can used to define instance-based access.
 *
 * grant: one or more operations (as a string or an array of strings)
 * to: (optional) one or more user roles the privilege is granted to
 * where: (optional) a condition that further restricts access
 *
 * example:
 *  { grant: ['READ','WRITE'], to: 'admin' }
    { grant: 'READ', where: 'buyer = $user' }
 */
public class Privilege {

	public static enum PredefinedGrant {
		/**
		 * grant: '*' means all events
		 */
		ALL("*"),

		/**
		 * grant. 'WRITE' means all standard events that have write semantic
		 */
		WRITE("WRITE");

		private final String grant;

		PredefinedGrant(String grant) {
			this.grant = grant;
		}

		public boolean is(String grant) {
			return Privilege.is(this.grant, grant);
		}

		@Override
		public String toString() {
			return grant;
		}
	}

	/**
	 * Predefined roles that are handled specifically
	 */
	public static enum PredefinedRole {
		/**
		 * to: 'any' means all roles are accepted
		 */
		ANY_USER("any"),

		/**
		 * to: 'authenticated-user' means all authenticated users are accepted
		 */
		AUTHENTICATED_USER("authenticated-user"),

		/**
		 * to: 'system-user' is for technical users
		 */
		SYSTEM_USER( "system-user");

		private final String role;

		PredefinedRole(String role) {
			this.role = role;
		}

		public boolean is(String role) {
			// TODO: are roles case insensitive ?
			return Privilege.is(this.role, role);
		}

		@Override
		public String toString() {
			return role;
		}
	}

	/**
	 *  field 'grant:'
	 */
	@JsonProperty("grant")
	private List<String> grants = new ArrayList<>();

	/**
	 * field 'to:'
	 */
	@JsonProperty("to")
	private List<String> roles = new ArrayList<>();

	/**
	 * field 'where:'
	 */
	@JsonProperty("where")
	private String whereCondition;

	/**
	 * field '_where:'
	 */
	@JsonProperty("_where")
	private String cxnWhereCondition;


	public List<String> getGrants() {
		return Collections.unmodifiableList(grants);
	}

	public Privilege addGrant(PredefinedGrant grant) {
		return addGrant(grant.toString());
	}

	public Privilege addGrant(String grant) {
		Objects.requireNonNull( StringUtils.notEmpty(grant) );

		grants.add(grant.trim());
		return this;
	}

	public List<String> getRoles() {
		return Collections.unmodifiableList(roles);
	}

	public Privilege addRole(PredefinedRole role) {
		return addRole(role.toString());
	}

	public Privilege addRole(String role) {
		Objects.requireNonNull( StringUtils.notEmpty(role)  );

		roles.add(role.trim());
		return this;
	}

	public String getWhereCondition() {
		return whereCondition;
	}

	public void setWhereCondition(String whereCondition) {
		this.whereCondition = whereCondition;
	}

	public String getCxnWhereCondition() {
		return cxnWhereCondition;
	}

	public void setCxnWhereCondition(String cxnWhereCondition) {
		this.cxnWhereCondition = cxnWhereCondition;
	}

	public static boolean is(String a, String b) {
		return a == b || (a != null && b != null && a.equals(b));
	}
}
