/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.mtx;

import java.time.Duration;
import java.util.List;
import java.util.Map;

import com.sap.cds.services.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.mtx.impl.Authenticator;
import com.sap.cds.mtx.impl.CacheParams;
import com.sap.cds.mtx.impl.ClientCredentialJwtAccess;
import com.sap.cds.mtx.impl.ClientCredentialJwtReader;
import com.sap.cds.mtx.impl.SidecarAccess;
import com.sap.cds.mtx.impl.SidecarAccessV1;
import com.sap.cds.mtx.impl.SidecarAccessV2;
import com.sap.cds.mtx.impl.XsuaaParams;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy.Sidecar.Cache;
import com.sap.cds.services.environment.ServiceBinding;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.XsuaaUtils;

/**
 * Utility class for the Sidecar
 */
public class MtxUtils {

	private static final Logger logger = LoggerFactory.getLogger(MtxUtils.class);
	private static final String BEARER_TOKEN_PREFIX = "Bearer ";

	private final CdsRuntime runtime;
	private final XsuaaUtils xsuaaUtils;

	public MtxUtils(CdsRuntime runtime) {
		this.runtime = runtime;
		this.xsuaaUtils = new XsuaaUtils(runtime);
	}

	private String getSidecarUrl() {
		MultiTenancy config = runtime.getEnvironment().getCdsProperties().getMultiTenancy();
		return config.getSidecar().getUrl();
	}

	public boolean dynamicDeployerEnabled() {
		return runtime.getEnvironment().getCdsProperties().getMultiTenancy().getDeployer().getUrl() != null;
	}

	public boolean mtxEnabled() {
		return !StringUtils.isEmpty(getSidecarUrl());
	}

	public SidecarAccess getSidecarAccess() {
		if (!mtxEnabled()) {
			return null;
		}
		ClientCredentialJwtReader jwtReader = createClientCredentialJwtReader();
		Authenticator authenticator;
		if (jwtReader == null) {
			logger.debug("Found no UAA binding: No JWT token can be returned");
			authenticator = Authenticator.NONE;
		} else {
			authenticator = new ClientCredentialJwtAccess(jwtReader);
		}
		if (runtime.getEnvironment().getCdsProperties().getMultiTenancy().getSidecar().getVersion() == 1) {
			return new SidecarAccessV1(getSidecarUrl(), authenticator);
		}
		return new SidecarAccessV2(getSidecarUrl(), authenticator);
	}

	public ClientCredentialJwtReader createClientCredentialJwtReader() {
		if (!mtxEnabled()) {
			return null;
		}
		List<ServiceBinding> uaaBindings = xsuaaUtils.getXsuaaServiceBindings();
		if (!uaaBindings.isEmpty()) {
			ServiceBinding uaaInstance = uaaBindings.get(0);
			return createClientCredentialJwtReader((uaaUrl, clientId, clientSecret) -> new ClientCredentialJwtReader(new XsuaaParams(uaaUrl, clientId, clientSecret)), uaaInstance.getCredentials());
		}
		// should only happen during testing
		return null;
	}

	public <T> T createClientCredentialJwtReader(ClientCredentialJwtCreator<T> creator, Map<String, Object> credentials) {
		String uaaUrl = (String) credentials.get("url");
		String clientId =  (String) credentials.get("clientid");
		String clientSecret =  (String) credentials.get("clientsecret");
		return creator.createClientCredentialJwtProvider(uaaUrl, clientId, clientSecret);
	}

	@FunctionalInterface
	public static interface ClientCredentialJwtCreator<T> {

		T createClientCredentialJwtProvider(String uaaUrl, String clientId, String clientSecret);

	}

	public CacheParams getCacheParams() {
		Cache cache = runtime.getEnvironment().getCdsProperties().getMultiTenancy().getSidecar().getCache();
		long maximumSize = cache.getMaxSize();
		long expirationDuration = cache.getExpirationTime();
		long refreshDuration = cache.getRefreshTime();

		return new CacheParams(maximumSize, Duration.ofSeconds(expirationDuration), Duration.ofSeconds(refreshDuration), false);
	}

	public String getBearerAuthorizationHeader(String jwt) {
		if (jwt == null || jwt.startsWith(BEARER_TOKEN_PREFIX)) {
			return jwt;
		}
		return BEARER_TOKEN_PREFIX + jwt;
	}

}
