/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.rest.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

import org.apache.http.Header;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Implementation of a simple JSON REST client.
 */
public class JsonRestClient {

	private static final Logger logger = LoggerFactory.getLogger(JsonRestClient.class);

	private static final String UTF_8 			= "UTF-8";
	private static final String AUTHORIZATION 	= "Authorization";
	private static final String BEARER 			= "Bearer";
	private static final String CLIENT_ID 		= "client_id";
	private static final String CLIENT_SECRET 	= "client_secret";
	private static final String GRANT_TYPE 		= "grant_type";
	private static final String JSON_CONTENT 	= "application/json";
	private static final String CONTENT_TYPE 	= "content-type";
	private static final String ACESS_TOKEN     = "access_token";
	private static final String APP_FORM_ENC    = "application/x-www-form-urlencoded";


	protected final ObjectMapper mapper = new ObjectMapper();
	private final JsonRestClientConfiguration restConfig;
	private final List<Header> additionalHeaders;

	private String oauthToken;

	// shared connection pool
	private final static CloseableHttpClient client = HttpClients.custom()
			.setConnectionManager(new PoolingHttpClientConnectionManager())
			.build();

	/**
	 * Creates the client and performs the client authentication.
	 *
	 * @param cfg client configuration
	 */
	public JsonRestClient(JsonRestClientConfiguration cfg) {
		this(cfg, null);
	}

	/**
	 * Creates the client and performs the client authentication.
	 *
	 * @param cfg client configuration
	 * @param additionalHeaders additional request headers for the auth request
	 */
	public JsonRestClient(JsonRestClientConfiguration cfg, List<Header> additionalHeaders) {
		this.restConfig = cfg;
		this.additionalHeaders = additionalHeaders;
	}

	/**
	 * Performs a HTTP GET request and returns the JSON response.
	 *
	 * @param path destination URL path
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode getRequest(String path) throws IOException {
		HttpGet get = new HttpGet(getUrl(path));
		try {
			return handleJsonResponse(handleRequest(get));
		} finally {
			get.releaseConnection();
		}
	}

	/**
	 * Performs a HTTP GET request and returns the response code only.
	 *
	 * @param path destination URL path
	 *
	 * @return response code
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 */
	public int getRequestWithOnlyResponseCode(String path) throws IOException {
		CloseableHttpResponse response = null;
		HttpGet get = new HttpGet(getUrl(path));
		try {
			response = handleRequest(get);
			return response.getStatusLine().getStatusCode();
		} finally {
			if (response != null) {
				response.close();
			}
			get.releaseConnection();
		}
	}

	/**
	 * Performs a HTTP DELETE request and returns the JSON response.
	 *
	 * @param path destination URL path
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode deleteRequest(String path) throws IOException {
		HttpDelete del = new HttpDelete(getUrl(path));
		try {
			return handleJsonResponse(handleRequest(del));
		} finally {
			del.releaseConnection();
		}
	}

	/**
	 * Performs a HTTP PUT request and returns the JSON response.
	 *
	 * @param path destination URL path
	 * @param data JSON request body
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode putRequest(String path, JsonNode data) throws IOException {
		HttpPut put = new HttpPut(getUrl(path));
		if (data != null) {
			put.setEntity(new StringEntity(mapper.writer().writeValueAsString(data), UTF_8));
		}
		try {
			return handleJsonResponse(handleRequest(put));
		} finally {
			put.releaseConnection();
		}

	}

	/**
	 * Performs a HTTP POST request and returns the JSON response.
	 *
	 * @param path destination URL path
	 * @param data JSON request body
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode postRequest(String path, JsonNode data) throws IOException {
		HttpPost post = new HttpPost(getUrl(path));
		if (data != null) {
			post.setEntity(new StringEntity(mapper.writer().writeValueAsString(data), UTF_8));
		}
		try {
			return handleJsonResponse(handleRequest(post));
		} finally {
			post.releaseConnection();
		}

	}

	/**
	 * Returns decoded JWT token used by the client.
	 *
	 * @return decoded JWT
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 */
	public JsonNode getJwtTokenInfo() throws IOException {

		String[] jwtParts = getOauthToken().split("\\.");
		if(jwtParts.length == 3) {
			String base64EncodedBody = jwtParts[1];
			String jwtInfo = new String(Base64.getUrlDecoder().decode(base64EncodedBody)); // ISO-8859-1 //NOSONAR

			try {
				return mapper.readValue(jwtInfo, JsonNode.class);
			} catch(Exception e) { // NOSONAR
				throw new IllegalStateException("Failed to parse JWT token!");
			}
		}
		throw new IllegalStateException("Failed to parse JWT token!");
	}


	private CloseableHttpResponse handleRequest(HttpRequestBase request) throws IOException {
		setAuthHeader(request);
		CloseableHttpResponse response = client.execute(request);
		if(response.getStatusLine().getStatusCode() == HttpStatus.SC_UNAUTHORIZED) {
			// retry with a refreshed token
			authenticate();
			setAuthHeader(request);
			response = client.execute(request);
		}
		return response;
	}

	/**
	 * Performs the OAuth2 authentication.
	 *
	 * @throws IOException
	 */
	private void authenticate() throws IOException {

		logger.info("Authenticating client '{}' with '{}' on '{}'", restConfig.getName(), restConfig.getGrantType(), restConfig.getTokenEndpoint());

		// create a post request
		HttpPost post = new HttpPost(restConfig.getTokenEndpoint());
		post.addHeader(CONTENT_TYPE, APP_FORM_ENC);

		// set additional headers if available
		if (additionalHeaders != null) {
			additionalHeaders.forEach(header -> post.addHeader(header));
		}

		// set authentication headers
		List<NameValuePair> params = new ArrayList<NameValuePair>(3);
		params.add(new BasicNameValuePair(GRANT_TYPE, restConfig.getGrantType()));
		params.add(new BasicNameValuePair(CLIENT_ID, restConfig.getClientId()));
		params.add(new BasicNameValuePair(CLIENT_SECRET, restConfig.getClientSecret()));

		// and finally request the authentication
		CloseableHttpResponse response = null;
		try {
			post.setEntity(new UrlEncodedFormEntity(params, UTF_8));
			response = client.execute(post);
			oauthToken =  handleJsonResponse(response).get(ACESS_TOKEN).asText().trim();
			logger.debug("Successfully authenticated the client '{}'", restConfig.getName());
		} finally {
			if (response != null) {
				response.close();
			}
			post.releaseConnection();
		}
	}

	/**
	 * Sets the authorization header of the HTTP request.
	 *
	 * @param request HTTP request
	 */
	private void setAuthHeader(HttpRequest request) throws IOException {

		String token = BEARER + " " + getOauthToken();

		logger.debug("Creating {} request to {} with token '{}'", request.getRequestLine().getMethod(), request.getRequestLine().getUri(), token);

		request.removeHeaders(CONTENT_TYPE);
		request.addHeader(CONTENT_TYPE, JSON_CONTENT);
		request.removeHeaders(AUTHORIZATION);
		request.addHeader(AUTHORIZATION, token);
	}


	private String getOauthToken() throws IOException {
		if (oauthToken == null) {
			try {
				// authenticate the API
				authenticate();
			} catch (IOException e) {
				logger.error("The rest client '{}' could not be authenticated!", restConfig.getName());
				throw e;
			}
		}
		return oauthToken;
	}

	/**
	 * Handles HTTP response and parses the JSON content.
	 *
	 */
	private JsonNode handleJsonResponse(CloseableHttpResponse response) throws IOException {
		try {
			int code = response.getStatusLine().getStatusCode();

			logger.debug("Responded with status code '{}'", code);

			if (code >= 200 && code <= 207) {
				String contentType = JSON_CONTENT;
				if (response.getEntity() != null) {
					if (response.getEntity().getContentType() != null) {
						contentType = response.getEntity().getContentType().getValue();
					}
					if (contentType.contains(JSON_CONTENT)) {
						String jsonData = EntityUtils.toString(response.getEntity());
						return mapper.readValue(jsonData, JsonNode.class);
					} else {
						throw new IOException("Unexpected response format: Expected JSON but found '" + contentType + "'");
					}
				} else {
					return mapper.readValue("{}", JsonNode.class);
				}
			} else {
				String reason = response.getStatusLine().getReasonPhrase();
				throw new JsonRestClientResponseException(code, "Unexpected request HTTP response (" + code + ") " + reason);
			}
		} finally {
			response.close();
		}
	}

	private String getUrl(String path) {
		return restConfig.getApiUrl() + path;
	}
}
