/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.utils.runtime;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.servlet.http.HttpServletRequest;

import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.HttpHeaders;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.QueryParameters;
import com.sap.cds.services.utils.TemporalRangeUtils;

/**
 * {@link ParameterInfo} instance derived from current HTTP request.
 */
public class HttpParameterInfo implements ParameterInfo {

	private final HttpServletRequest httpRequest;
	private final LocaleUtils localeUtils;

	private Locale locale;
	private Instant validFrom;
	private Instant validTo;

	private HttpParameterInfo(HttpServletRequest httpRequest, LocaleUtils localeUtils) {
		this.httpRequest = httpRequest;
		this.localeUtils = localeUtils;
	}

	/**
	 * Creates the {@link ParameterInfo} instance based on the current servlet (http) request.
	 * @param httpRequest	The http request
	 * @param runtime		The {@link CdsRuntime}
	 *
	 * @return	The {@link ParameterInfo} instance or {@code null} if there is no http request scope
	 */
	public static ParameterInfo fromRequest(HttpServletRequest httpRequest, CdsRuntime runtime) {
		return new HttpParameterInfo(httpRequest, new LocaleUtils(runtime.getEnvironment().getCdsProperties()));
	}

	@Override
	public Map<String, String> getHeaders() {
		Map<String, String> result = new HashMap<>();

		Enumeration<String> headerKeys = httpRequest.getHeaderNames();
		if (headerKeys == null) {
			return result;
		}
		while (headerKeys.hasMoreElements()) {
			String headerKey = headerKeys.nextElement();
			result.put(headerKey, getHeader(headerKey));
		}

		return result;
	}

	@Override
	public String getHeader(String id) {
		// there might be several headers with the same id!
		Enumeration<String> headerValues = httpRequest.getHeaders(id);
		if (headerValues == null || !headerValues.hasMoreElements()) {
			return null;
		}
		return Collections.list(headerValues).stream().collect(Collectors.joining(","));
	}

	@Override
	public String getQueryParameter(String key) {
		String[] values = httpRequest.getParameterMap().get(key);
		if (values != null && values.length >= 1) {
			return values[0];
		}
		return null;
	}

	@Override
	public Stream<String> getQueryParameters() {
		return httpRequest.getParameterMap().keySet().stream();
	}

	@Override
	public Map<String, String> getQueryParams() {
		Map<String, String[]> parametersMap = httpRequest.getParameterMap();

		Map<String, String> result = new HashMap<>();
		for (Map.Entry<String, String[]> entry : parametersMap.entrySet()) {
			String[] values = entry.getValue();
			String valueString = values != null ? Arrays.asList(values).stream().collect(Collectors.joining(",")) : null;
			result.put(entry.getKey(), valueString);
		}
		return result;
	}

	@Override
	public Locale getLocale() {
		if (this.locale == null) {
			this.locale = localeUtils.getLocale(
					getQueryParameter(QueryParameters.SAP_LANGUAGE),
					getHeader(HttpHeaders.X_SAP_REQUEST_LANGUAGE_HEADER),
					getHeader(HttpHeaders.ACCEPT_LANGUAGE));
		}
		return locale;
	}

	@Override
	public Instant getValidFrom() {
		if (this.validFrom == null) {
			initTemporalRange();
		}
		return this.validFrom;
	}

	@Override
	public Instant getValidTo() {
		if (this.validTo == null) {
			initTemporalRange();
		}
		return this.validTo;
	}

	private void initTemporalRange() {
		Instant[] ranges = TemporalRangeUtils.getTemporalRanges(
				getQueryParameter(QueryParameters.VALID_FROM),
				getQueryParameter(QueryParameters.VALID_TO),
				getQueryParameter(QueryParameters.VALID_AT));

		this.validFrom = ranges[0];
		this.validTo = ranges[1];
	}

}

