/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cds.services.utils.messaging.jms;

import com.sap.cds.services.utils.messaging.jms.MessageEmitter;
import com.sap.cds.services.utils.messaging.jms.MessageQueueReader;
import com.sap.cds.services.utils.messaging.jms.TopicAccessor;
import com.sap.cds.services.utils.messaging.service.MessagingBrokerQueueListener;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.JMSException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BrokerConnection {
    private static final Logger logger = LoggerFactory.getLogger(BrokerConnection.class);
    private static final Timer REESTABLISH_TIMER = new Timer("Broker Connection Reestablisher", true);
    private static final int MIN_REESTABLISH_DELAY = 120000;
    private static final int MAX_REESTABLISH_DELAY = 600000;
    private final String name;
    private final ConnectionFactory connectionFactory;
    private final List<MessageQueueReader> queueReaders = new ArrayList<MessageQueueReader>();
    private volatile Connection connection;
    private volatile MessageEmitter emitter;
    private volatile boolean isConnected;
    private volatile int reestablishDelay = 120000;

    public BrokerConnection(String name, ConnectionFactory connectionFactory) {
        this.name = name;
        this.connectionFactory = connectionFactory;
    }

    public void connect() throws IOException {
        logger.debug("Opening broker connection '{}'", (Object)this.name);
        if (!this.isConnected) {
            AtomicBoolean preventScheduling = new AtomicBoolean();
            try {
                this.connection = this.connectionFactory.createConnection();
                this.connection.setExceptionListener(e -> {
                    logger.warn("The broker connection '{}' failed", (Object)this.name);
                    if (!preventScheduling.getAndSet(true)) {
                        this.scheduleReconnect();
                    }
                });
                this.emitter = new MessageEmitter(this.connection);
                this.connection.start();
                this.isConnected = true;
                this.reestablishDelay = 120000;
                logger.info("The messaging broker connection '{}' has been established", (Object)this.name);
            }
            catch (JMSException e2) {
                if (!preventScheduling.getAndSet(true)) {
                    this.scheduleReconnect();
                }
                throw new IOException("Could not establish broker connection '" + this.name + "'", e2);
            }
        } else {
            logger.warn("The broker connection '{}' is already connected", (Object)this.name);
        }
    }

    public void close() throws JMSException {
        try {
            if (this.connection != null) {
                logger.debug("Closing broker connection '{}'", (Object)this.name);
                this.connection.close();
            }
        }
        finally {
            this.isConnected = false;
        }
    }

    private void scheduleReconnect() {
        logger.debug("The broker connection '{}' reconnect attempt will be scheduled in {} ms", (Object)this.name, (Object)this.reestablishDelay);
        REESTABLISH_TIMER.schedule(new TimerTask(){

            @Override
            public void run() {
                try {
                    BrokerConnection.this.close();
                }
                catch (JMSException e) {
                    logger.debug("Failed to close broker connection '{}' before reconnecting", (Object)BrokerConnection.this.name, (Object)e);
                }
                try {
                    BrokerConnection.this.connect();
                    for (MessageQueueReader queueReader : BrokerConnection.this.queueReaders) {
                        queueReader.startListening(BrokerConnection.this.connection);
                    }
                }
                catch (IOException | JMSException e) {
                    logger.error("Failed to reestablish broker connection '{}'", (Object)BrokerConnection.this.name, (Object)e);
                }
            }
        }, this.reestablishDelay);
        this.reestablishDelay = Math.min(this.reestablishDelay * 2, 600000);
    }

    public void registerQueueListener(String queue, MessagingBrokerQueueListener listener, TopicAccessor topicAccessor, int maxFailedCount) throws IOException {
        try {
            logger.debug("Registering queue listener on '{}'", (Object)queue);
            MessageQueueReader queueReader = new MessageQueueReader(queue, listener, topicAccessor, maxFailedCount);
            this.queueReaders.add(queueReader);
            queueReader.startListening(this.connection);
        }
        catch (JMSException e) {
            throw new IOException(e.getMessage(), e);
        }
    }

    public void emitTopicMessage(String topic, String message) {
        logger.debug("Emitting a message to topic '{}'", (Object)topic);
        this.emitter.emitTopicMessage(topic, message);
    }

    public String getName() {
        return this.name;
    }

    public boolean isConnected() {
        return this.isConnected;
    }
}

